// --------------------------------
// <copyright file="TraceLogger.cs" company="Genesys Telecommunications Laboratories">
//
//  Any authorized distribution of any copy of this code (including any related 
//  documentation) must reproduce the following restrictions, disclaimer and copyright 
//  notice:
//  The Genesys Name, trademarks and/or logo(s) of Genesys shall not be used to Name 
//  (even as a part of another Name), endorse and/or promote products derived from 
//  this code without prior written permission from Genesys Telecommunications 
//  Laboratories, Inc. 
//  
//  The use, copy, and/or distribution of this code is subject to the terms of the Genesys 
//  Developer License Agreement.  This code shall not be used, copied, and/or 
//  distributed under any other license agreement.    
//  THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. 
//  (\'GENESYS\') 'AS IS' WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY 
//  DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND 
//  WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT 
//  NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
//  PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL 
//  NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO 
//  EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, 
//  CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT 
//  NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
//  Copyright (c) 2008-2009 Genesys Telecommunications Laboratories, Inc. All Rights Reserved.
//
// </copyright>
// ---------------------------------

namespace YourNamespace
{
	using System;
	using System.Diagnostics;
	using Genesyslab.Platform.Commons.Logging;

	/// <summary>
	///	This is the default implementation of the Logger that sends messages to debug output stream
	/// </summary>
	public class TraceLogger: ILogger
	{
		/// <summary>
		///  Typecode for debugging messages. 
		/// </summary>
		public const int LevelDebug = 0;

		/// <summary>
		///  String Name for debug level messages.
		/// </summary>
		public const string LevelDebugName = "DEBUG";

		/// <summary>
		/// Typecode for informational messages. 
		/// </summary>
		public const int LevelInfo = 1;

		/// <summary>
		///  String Name for info level messages.
		/// </summary>
		public const string LevelInfoName = "INFO";

		/// <summary>
		/// Typecode for warning messages.
		/// </summary>
		public const int LevelWarn = 2;

		/// <summary>
		///  String Name for warn level messages.
		/// </summary>
		public const string LevelWarnName = "WARN";

		/// <summary>
		/// Typecode for error messages.
		/// </summary>
		public const int LevelError = 3;

		/// <summary>
		///  String Name for error level messages.
		/// </summary>
		public const string LevelErrorName = "ERROR";

		/// <summary>
		/// Typecode for fatal error messages.
		/// </summary>
		public const int LevelFatal = 4;

		/// <summary>
		///  String Name for fatal error level messages.
		/// </summary>
		public const string LevelFatalName = "FATAL ERROR";

		/// <summary>
		/// Typecode for disabled log levels.
		/// </summary>
		public const int LevelDisabled = 5;

		private int logLevel;

		/// <summary>
		/// Creates a new TraceLogger with the priority set to DEBUG.
		/// </summary>
		public TraceLogger(): this(LevelDebug)
		{
		}

		/// <summary>
		/// Creates a new TraceLogger.
		/// </summary>
		/// <param Name="logLevel">The Log level typecode.</param>
		public TraceLogger(int logLevel)
		{
			this.logLevel = logLevel;
		}

		/// <summary>
		/// Logs a debug message.
		/// </summary>
		/// <param Name="message">The Message</param>
		public void Debug(object message)
		{
			Debug(message, null as Exception);
		}

		/// <summary>
		/// Logs a debug message. 
		/// </summary>
		/// <param Name="message">The Message</param>
		/// <param Name="exception">The Exception</param>
		public void Debug(object message, Exception exception)
		{
			Log(TraceLogger.LevelDebug, TraceLogger.LevelDebugName, message, exception); 
		}

		/// <summary>
		/// Logs a debug message.
		/// </summary>
		/// <param Name="format">Message format</param>
		/// <param Name="args">Array of objects to write using format</param>
		public void DebugFormat( string format, params Object[] args )
		{
			Debug(String.Format(format, args));
		}

		/// <summary>
		/// Determines if messages of priority "debug" will be logged.
		/// </summary>
		/// <value>True if "debug" messages will be logged.</value> 
		public bool IsDebugEnabled
		{
			get
			{
				return (logLevel <= LevelDebug);
			}
		}

		/// <summary>
		/// Logs an info message.
		/// </summary>
		/// <param Name="message">The Message</param>
		public void Info( object message )
		{
			Info(message, null as Exception);
		}

		/// <summary>
		/// Logs an info message. 
		/// </summary>
		/// <param Name="message">The Message</param>
		/// <param Name="exception">The Exception</param>
		public void Info( object message, Exception exception)
		{
			Log(TraceLogger.LevelInfo, TraceLogger.LevelInfoName, message, exception); 
		}

		/// <summary>
		/// Logs an info message.
		/// </summary>
		/// <param Name="format">Message format</param>
		/// <param Name="args">Array of objects to write using format</param>
		public void InfoFormat( string format, params Object[] args )
		{
			Info(String.Format(format, args));
		}

		/// <summary>
		/// Determines if messages of priority "info" will be logged.
		/// </summary>
		/// <value>True if "info" messages will be logged.</value>
		public bool IsInfoEnabled
		{
			get
			{
				return (logLevel <= LevelInfo);
			}
		}

		/// <summary>
		/// Logs a warn message.
		/// </summary>
		/// <param Name="message">The Message</param>
		public void Warn(object message )
		{
			Warn(message, null as Exception);
		}

		/// <summary>
		/// Logs a warn message. 
		/// </summary>
		/// <param Name="message">The Message</param>
		/// <param Name="exception">The Exception</param>
		public void Warn(object message, Exception exception)
		{
			Log(TraceLogger.LevelWarn, TraceLogger.LevelWarnName, message, exception); 
		}

		/// <summary>
		/// Logs an warn message.
		/// </summary>
		/// <param Name="format">Message format</param>
		/// <param Name="args">Array of objects to write using format</param>
		public void WarnFormat( string format, params Object[] args )
		{
			Warn(String.Format(format, args));
		}

		/// <summary>
		/// Determines if messages of priority "warn" will be logged.
		/// </summary>
		/// <value>True if "warn" messages will be logged.</value>
		public bool IsWarnEnabled
		{
			get
			{
				return (logLevel <= LevelWarn);
			}
		}

		/// <summary>
		/// Logs an error message.
		/// </summary>
		/// <param Name="message">The Message</param>
		public void Error(object message )
		{
			Error(message, null as Exception);
		}

		/// <summary>
		/// Logs an error message. 
		/// </summary>
		/// <param Name="message">The Message</param>
		/// <param Name="exception">The Exception</param>
		public void Error(object message, Exception exception)
		{
			Log(TraceLogger.LevelError, TraceLogger.LevelErrorName, message, exception); 
		}
		/// <summary>
		/// Logs an error message.
		/// </summary>
		/// <param Name="format">Message format</param>
		/// <param Name="args">Array of objects to write using format</param>
		public void ErrorFormat( string format, params Object[] args )
		{
			Error(String.Format(format, args));
		}

		/// <summary>
		/// Determines if messages of priority "error" will be logged.
		/// </summary>
		/// <value>True if "error" messages will be logged.</value>
		public bool IsErrorEnabled
		{
			get
			{
				return (logLevel <= LevelError);
			}
		}

		/// <summary>
		/// Logs a fatal error message.
		/// </summary>
		/// <param Name="message">The Message</param>
		public void FatalError(object message )
		{
			FatalError(message, null as Exception);
		}

		/// <summary>
		/// Logs a fatal error message.
		/// </summary>
		/// <param Name="message">The Message</param>
		/// <param Name="exception">The Exception</param>
		public void FatalError(object message, Exception exception)
		{
			Log(TraceLogger.LevelFatal, TraceLogger.LevelFatalName, message, exception); 
		}

		/// <summary>
		/// Logs a fatal error message.
		/// </summary>
		/// <param Name="format">Message format</param>
		/// <param Name="args">Array of objects to write using format</param>
		public void FatalErrorFormat( string format, params Object[] args )
		{
			FatalError(String.Format(format, args));
		}

		/// <summary>
		/// Determines if messages of priority "fatalError" will be logged.
		/// </summary>
		/// <value>True if "fatalError" messages will be logged.</value>
		public bool IsFatalErrorEnabled
		{
			get 
			{
				return (logLevel <= LevelFatal); 
			}
		}

		/// <summary>
		/// A Common method to log.
		/// </summary>
		/// <param Name="level">The level of logging</param>
		/// <param Name="levelName">The Level Name</param>
		/// <param Name="message">The Message</param>
		/// <param Name="exception">The Exception</param>
		protected void Log(int level, string levelName, object message, Exception exception) 
		{
			if(logLevel <= level)
			{
				System.Diagnostics.Debug.WriteLine(string.Format("[{0}] {1}", levelName, message));
				
				if(exception != null)
				{
					System.Diagnostics.Debug.WriteLine(exception.StackTrace);
				}
			}
		}

		/// <summary>
		///	Just returns this logger (<c>TraceLogger</c> is not hierarchical).
		/// </summary>
		/// <param Name="Name">Ignored</param>
		/// <returns>This ILogger instance.</returns> 
		public ILogger CreateChildLogger(string name )
		{
			return this;
		}
	}
}	
