/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2007 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/

package agent.interaction.samples;

import java.awt.event.ActionEvent;
import java.util.HashMap;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import com.genesyslab.ail.Dn;
import com.genesyslab.ail.InteractionVoice;
import com.genesyslab.ail.Switch;
import com.genesyslab.ail.event.InteractionEvent;

/**
 * This example uses a GUI application to show how to transfer a call and how
 * to set up a three-way conference call. <br/><br/>
 * @version 7.5.001.02
 *
 */
public class MultipartyVoiceInteraction extends SimpleVoiceInteraction {

	JLabel multipartyVoiceTargetDnLabel;
	JTextField multipartyVoiceTargetDnText;
	JTextField multipartyVoiceReasonText;

	JButton conferenceButton;
	JButton transferButton;
	JButton completeButton;
	JRadioButton singleStep;
	JRadioButton dualStep;
	JRadioButton muteTransfer;

	boolean thisIsAConferenceCall = true;

	boolean switchCanDoSingleStep = false;
	boolean switchCanDoMuteTransfer = false;
	boolean switchCanDoDualStep = false;

	/**
	 * Creates an instance of MultipartyVoiceInteraction.<br/>
	 *
	 * @param windowTitle
	 *            The title of the GUI window for this example.
	 */
	public MultipartyVoiceInteraction(String windowTitle) {
		super(windowTitle);
	}

	/**
	 * Tells the AgentInteractionGui class which example has called
	 * the GUI, so that it can determine which widgets will be enabled when the
	 * GUI is created.
	 */
	public void setSampleType() {
		// Let the GUI class know what type of example this is
		sampleType = "MultipartyVoiceInteraction";
	}
	/**
	 * Links widgets (user interface
	 * components) to API functionality and registers a listener for
	 * events on the Place the agent has logged in to.
	 * <br/><br/>
	 * The linked widgets for Multiparty Voice are buttons for starting a
	 * conference or a transfer, and to select how to process the action
	 * (single step, mute, dual step) using radio buttons.
	 * @see SimplePlace#linkWidgetsToGui()
	 */
	public void linkWidgetsToGui() {
		// Execute the corresponding method in SimpleVoiceInteraction first
		super.linkWidgetsToGui();

		// Link these fields to the corresponding GUI components
		multipartyVoiceTargetDnLabel = agentInteractionGui.multipartyVoiceTargetDnLabel;
		multipartyVoiceTargetDnText = agentInteractionGui.multipartyVoiceTargetDnTextField;
		multipartyVoiceReasonText = agentInteractionGui.multipartyVoiceReasonTextField;

		singleStep = agentInteractionGui.singleStepRadioButton;
		dualStep = agentInteractionGui.dualStepRadioButton;
		muteTransfer = agentInteractionGui.muteTransferRadioButton;


		conferenceButton = agentInteractionGui.conferenceButton;
		conferenceButton.setAction(new AbstractAction("Conference") {
			public void actionPerformed(ActionEvent event) {
				thisIsAConferenceCall = true;
				performConference();

			}
		});

		transferButton = agentInteractionGui.transferButton;
		transferButton.setAction(new AbstractAction("Transfer") {
			public void actionPerformed(ActionEvent event) {
				thisIsAConferenceCall = false;
				performTransfer();
			}
		});

		completeButton = agentInteractionGui.completeButton;
		completeButton.setAction(new AbstractAction("Complete") {
			public void actionPerformed(ActionEvent event) {
				performComplete();
			}
		});

		//By default, AgentInteractioGui sets action to toggle buttons
		//so that a single radio button is selected at a certain time.
		//Selecting single step transfer
		singleStep.setSelected(true);

		//To determine whether or not the user is allowed to
		// use the Transfer and Conference modes, the example
		// tests whether the
		// underlying switch is capable to perform them.

		if(sampleDn instanceof Dn )
		{
			Switch theSwitch = sampleDn.getSwitch();
			if (theSwitch != null) {

				switchCanDoSingleStep = theSwitch.isCapable(InteractionVoice.Action.SINGLE_STEP_TRANSFER)
					|| theSwitch.isCapable(InteractionVoice.Action.SINGLE_STEP_CONFERENCE);
				switchCanDoMuteTransfer = theSwitch.isCapable(InteractionVoice.Action.MUTE_TRANSFER);
				switchCanDoDualStep = theSwitch.isCapable(InteractionVoice.Action.INIT_TRANSFER)
					|| theSwitch.isCapable(InteractionVoice.Action.CONFERENCE);
			}
		}
		singleStep.setVisible(switchCanDoSingleStep);
		muteTransfer.setVisible(switchCanDoMuteTransfer);
		dualStep.setVisible(switchCanDoDualStep);

	}
	/**
     * This class receives InteractionEvents, which are handled by the
     * handleInteractionEvent() method. This class also has widgets whose state
     * must be set when it receives these events.<br/><br/>
     *
     * This method enables or disables the specified widgets, which are normally
     * just buttons and radio buttons, but may include other widgets. To do
     * this, it use the isPossible() method of the Possible interface.
     *
     * The isPossible() method returns either true or false, depending on
     * whether a particular action is possible. This boolean value is used to
     * determine whether the widget will be enabled.
     */
	public void setInteractionWidgetState() {

		super.setInteractionWidgetState();

		if (sampleInteraction instanceof InteractionVoice) {

			//The Conference button should be enabled if at least one type
			// of conference is available:
			// single step OR dual step
			boolean conference = sampleInteraction.isPossible(InteractionVoice.Action.CONFERENCE)
				|| sampleInteraction.isPossible(InteractionVoice.Action.SINGLE_STEP_CONFERENCE);
			conferenceButton.setEnabled(conference);

			//The transfer button should be enabled if at least one type
			// of transfer is available:
			// single step OR mute OR dual step
			boolean transfer = (sampleInteraction
					.isPossible(InteractionVoice.Action.SINGLE_STEP_TRANSFER))
					|| (sampleInteraction.isPossible(InteractionVoice.Action.MUTE_TRANSFER))
					|| (sampleInteraction.isPossible(InteractionVoice.Action.INIT_TRANSFER));
			transferButton.setEnabled(transfer);

			//The complete button should be enabled if a transfer or a conference
			//is completed
			boolean complete = sampleInteraction.isPossible(InteractionVoice.Action.COMPLETE_TRANSFER)
				|| sampleInteraction.isPossible(InteractionVoice.Action.COMPLETE_CONFERENCE);
			completeButton.setEnabled(complete);

			//A type of transfer or conference can be selected
			//if this transfer or conference is possible
			boolean single = sampleInteraction.isPossible(InteractionVoice.Action.SINGLE_STEP_TRANSFER)
				|| sampleInteraction.isPossible(InteractionVoice.Action.SINGLE_STEP_CONFERENCE);
			singleStep.setEnabled(single);

			boolean dual = sampleInteraction.isPossible(InteractionVoice.Action.INIT_TRANSFER)
				|| sampleInteraction.isPossible(InteractionVoice.Action.CONFERENCE);
			dualStep.setEnabled(dual);

			muteTransfer.setEnabled(sampleInteraction.isPossible(InteractionVoice.Action.MUTE_TRANSFER));


		} else {
			conferenceButton.setEnabled(false);
			transferButton.setEnabled(false);
			completeButton.setEnabled(false);
			singleStep.setEnabled(false);
			muteTransfer.setEnabled(false);
			dualStep.setEnabled(false);
		}
	}

	/**
	 * Gets the user input corresponding to the transfer target/
	 */
	String getTransferTarget() {
		return multipartyVoiceTargetDnText.getText();
	}

	/**
	 * Gets the text input corresponding to the reason of the call.
	 * transfer
	 */
	String getTransferAgentReason() {
		return multipartyVoiceReasonText.getText();
	}
	/**
	 * Performs the Conference action associated with the Conference button.
	 * It takes into account the type of conference selected by the user
	 * (using toggle buttons).
	 */
	public void performConference(){
		try {
			// If "Single step" is selected...
			if (singleStep.isSelected() && sampleInteraction != null) {
				// ...a single step conference is required
				sampleInteraction.singleStepConference(getTransferTarget(), null, null,null, null);

				// If "Dual step" is selected...
			} else if (dualStep.isSelected() && sampleInteraction != null) {
				// ...a dual step conference is required

				sampleInteraction.initiateConference(getTransferTarget(), null, null,
						null, null);

				// If "Mute" is selected...
			} else if (muteTransfer.isSelected() && sampleInteraction != null) {
				// ...by default, a single step conference is performed

				sampleInteraction.singleStepConference(getTransferTarget(), null, null,null, null);
			}


		} catch (Exception _e) {
			_e.printStackTrace();
		}
	}
	/**
	 * Performs the Transfer action associated with the Transfer button.
	 * It takes into account the type of
	 * transfer selected by the user.
	 */

	public void performTransfer() {
		try {
			// If "Single step" is selected...
			if (singleStep.isSelected() && sampleInteraction != null) {
				// ...a single step transfer is required

				sampleInteraction.singleStepTransfer(getTransferTarget(), null,
						getTransferAgentReason(), null, null, null);

				// If "Dual step" is selected...
			} else if (dualStep.isSelected() && sampleInteraction != null) {
				// ...a dual step transfer is required
				sampleInteraction.initiateTransfer(getTransferTarget(), null,
						getTransferAgentReason(), null, null, null);

				// If "Mute" is selected...
			} else if (muteTransfer.isSelected() && sampleInteraction != null) {
				// ...a mute transfer is required

				sampleInteraction.muteTransfer(getTransferTarget(), null,
						getTransferAgentReason(), null, null, null);

			}
		} catch (Exception _e) {
			_e.printStackTrace();
		}
	}

	/**
	 * Performs the Complete action associated to the Complete button.
	 * It takes into account whether the action completes a transfer or a conference.
	 */
	public void performComplete() {
		try {
			if (thisIsAConferenceCall) {
				sampleInteraction.completeConference(null, null);
			} else {
				sampleInteraction.completeTransfer(null, null);
			}
		} catch (Exception _e) {
			_e.printStackTrace();
		}

	}

	/**
	 *
	 * Receives InteractionEvents, which report changes to the status or data
	 * for all interactions of this place.
	 * If you implement the PlaceListener interface, you must
	 * implement this method. <br/><br/>
	 *
	 * The MultipartyVoiceInteraction code example uses this method to log these
	 * InteractionEvents as they occur and to set the state of the widgets
	 * used by the MultipartyVoiceInteraction example.
	 */
	public void handleInteractionEvent(InteractionEvent event) {
		super.handleInteractionEvent(event);

		if(event.getInteraction().getId() == sampleInteraction.getId())
		{
			HashMap map = (HashMap) event.getExtensions();
			String info ="";
			if(map.containsKey(InteractionEvent.Extension.RINGING_TRANSFER_REASON))
			{
				info += "Transferred ("
					+ ((String)map.get(InteractionEvent.Extension.RINGING_TRANSFER_REASON))+")";
				multipartyVoiceReasonText.setText(info);
			}
			else if(map.containsKey(InteractionEvent.Extension.RINGING_CONFERENCE_REASON))
			{
				info += "Conference ("
					+(String) map.get(InteractionEvent.Extension.RINGING_CONFERENCE_REASON)+")";
				multipartyVoiceReasonText.setText(info);
			}
		}
	}
	/**
	 * This method is used to run the MultipartyVoiceInteraction example.
	 * @param args There are no arguments for this class.
	 */
	public static void main(String[] args) {
		//Schedule a job for the event-dispatching thread:
		//creating and showing this application's GUI.
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				MultipartyVoiceInteraction application = new MultipartyVoiceInteraction(
				"Agent Interaction (Java): Multiparty Voice Interaction");
			}
		});
	}

}