/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 20072008 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
package agent.interaction.samples;

import java.awt.event.ActionEvent;
import java.util.Collection;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.SwingUtilities;


import com.genesyslab.ail.Agent;
import com.genesyslab.ail.Dn;
import com.genesyslab.ail.Interaction;
import com.genesyslab.ail.Media;
import com.genesyslab.ail.Place;
import com.genesyslab.ail.event.DnEvent;
import com.genesyslab.ail.event.InteractionEvent;
import com.genesyslab.ail.event.PlaceEvent;
import com.genesyslab.ail.event.PlaceEventMediaAdded;
import com.genesyslab.ail.event.PlaceEventMediaRemoved;
import com.genesyslab.ail.event.PlaceEventMediaStatusChanged;
import com.genesyslab.ail.event.PlaceListener;

/**
 * This example uses a GUI application to show how to log an agent into a place
 * and queue. Once you have done this, the example enables you to set the agent
 * status to Ready, Not Ready, and After Call Work. It also lets you log the
 * agent out.<br/><br/>
 *
 * This example is also the base class for the other Agent Interaction (Java)
 * code examples, including SimpleVoiceInteraction and MultipartyVoiceInteraction.
 * <br/><br/>
 *
 * <strong>PLEASE NOTE:</strong> This code example will not work unless you edit
 * the configuration data in AgentInteractionData.java so that it matches objects
 * that already exist in your local configuration layer. <br/><br/>
 *
 * There are several steps needed to prepare an application to use the services
 * of Agent Interaction (Java). This example uses the services of other classes
 * to carry out these functions. <br/>
 *
 * <ul>
 * <li>AgentInteractionData makes various configuration data available to
 * the Agent Interaction (Java) code examples<br/></li>
 * <li>AgentInteractionGui creates the common graphical user interface
 * used by the code examples<br/></li>
 * <li>{@link Connector} connects the code examples to the Agent Interaction Layer
 * </li>
 * </ul>
 * @version 7.5.001.02
 */
public class SimplePlace implements PlaceListener {

    AgentInteractionData agentInteractionData;
    

    // Agent, place, and DN for the AIL examples
    /** Agent object used in this example. */
    public Agent sampleAgent;
    /** Place object used in this example. */
    public Place samplePlace;
    /** Dn object used in this example. */
    public Dn sampleDn;
    /** Media object used for managing e-mail in this example. */
    public Media sampleEmail;
    /** Media object used for managing open media in this example. */
    public Media sampleOpenMedia;
    /** Media object used for managing open media in this example. */
    public Media sampleChat;

    String workitem ="sms";

    String sampleType = "SimplePlace";
    JButton loginButton;
    JButton logoutButton;
    JButton readyButton;
    JButton notReadyButton;
    JButton afterCallWorkButton;

    JLabel loginNameLabel;
    JLabel dnLabel;
    JLabel agentPlaceLabel;
    JLabel agentQueueLabel;
    Collection mediaList;

    String dnEventStyle = "dnEvent";
    String placeEventStyle = "placeEvent";
    String interactionEventStyle = "interactionEvent";
    String errorStyle = "errorEvent";
    AgentInteractionGui agentInteractionGui;

    boolean voice = true;
    /**
     * Creates an instance of SimplePlace.
     *
     * The Connector class connects to the Agent Interaction Layer using the
     * AilLoader and AilFactory classes.
     *
     * The AgentInteractionData class supplies data from the Configuration Layer
     * for things like host, port, place, DN and agent attributes.
     *
     * After connecting, this method sets up an agent, place and DN for use by
     * the examples. Then it creates the GUI, links the GUI components (widgets)
     * to API functionality, and starts the application.
     *
     * @param windowTitle
     *            The title of the GUI window for this example.
     */
    public SimplePlace(String windowTitle) {

        setSampleType();
          	
        // Connect to AIL
        SimpleConnector connector = new SimpleConnector();
        
        // Make configuration layer data available to the examples
        agentInteractionData = connector.agentInteractionData;

       // Set up an agent, place and DN for use by the examples
        sampleAgent = (Agent) connector.ailFactory
                .getPerson(agentInteractionData.getAgent1UserName());
        samplePlace = connector.ailFactory.getPlace(agentInteractionData
                .getPlace1());
     
        sampleDn = connector.ailFactory.getDn(agentInteractionData.getDn1());      
        
        
        // Create the GUI
        agentInteractionGui = new AgentInteractionGui(windowTitle, sampleType);
        // Link the GUI components to API functionality
        linkWidgetsToGui();

        // Start the application
        agentInteractionGui.mainGuiWindow
                .setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        agentInteractionGui.mainGuiWindow.addWindowListener(connector);
        agentInteractionGui.mainGuiWindow.pack();
        agentInteractionGui.mainGuiWindow.setVisible(true);
        
        //guiReady = true;
    }

    /**
     * The AgentInteractionGui class creates a single graphical user interface
     * that is used by each of the Agent Interaction (Java) code examples.
     * This GUI contains all of the widgets used by the examples, but only the
     * widgets used by a particular example should be enabled. <br/><br/>
     *
     * This method tells the AgentInteractionGui class which example has called
     * the GUI, so that it can determine which widgets will be enabled when the
     * GUI is created.
     */
    protected void setSampleType() {
        // Let the GUI class know what type of example this is
        sampleType = "SimplePlace";
    }

    /**
     * This method does two important things: first, it links widgets (user interface
     * components) to API functionality, and second, it registers a listener for
     * events on the Place the agent has logged in to.
     * <br/><br/>
     *
     * The AgentInteractionGui class creates a single graphical user interface
     * that is used by each of the Agent Interaction (Java) code examples.
     * This GUI contains all of the widgets used by the examples, but they are
     * only placeholders. That is, they don't have any functionality when they
     * are created in the GUI class. <br/><br/>
     *
     * When a example is run, the linkWidgetsToGui() method is used to
     * associate these widgets with the appropriate fields in the example class,
     * as shown here: <br/><br/>
     *
     * <code>loginButton = agentInteractionGui.loginButton;</code> <br/><br/>
     *
     * In many cases, particularly for buttons and radio buttons, these fields
     * are then linked to actions that correspond to the functions they are
     * designed to carry out. For example:<br/><br/>
     *
     * <code>loginButton.setAction(new AbstractAction("Log In") {<br/>
     *     &nbsp;public void actionPerformed(ActionEvent actionEvent) {<br/>
     *         &nbsp;&nbsp;try {<br/>
     *             &nbsp;&nbsp;&nbsp;sampleAgent.login(samplePlace, agentInteractionData<br/>
     *                      &nbsp;&nbsp;&nbsp;&nbsp;.getLoginId1(), agentInteractionData.getPassword1(),<br/>
     *                      &nbsp;&nbsp;&nbsp;&nbsp;agentInteractionData.getQueue(), null, null, null);<br/>
     *     ...</code><br/><br/>
     *
     * As mentioned above, this method also contains the statement that registers
     * a listener for events on the Place the agent is logged in to:<br/><br/>
     *
     * <code>samplePlace.addPlaceListener(this);</code>
     */
    public void linkWidgetsToGui() {
        // Add a "Log In" button
        loginButton = agentInteractionGui.loginButton;
        loginButton.setAction(new AbstractAction("Log In") {
            public void actionPerformed(ActionEvent actionEvent) {
                try {
                	if(voice)                	
                		// Perform a voice-only login                	
                		samplePlace.login(agentInteractionData.getLoginId1(),
                            agentInteractionData.getPassword1(),
                            agentInteractionData.getQueue(), null, null, null);
                	else if(mediaList != null)
                		// Perform a multimedia login (this login is for all
                		// media types other than voice)
                		samplePlace.loginMultimedia( sampleAgent, mediaList, null, null);

                } catch (Exception exception) {
                    agentInteractionGui.writeLogMessage(exception.getMessage()
                    		+ "cause: " + exception.getCause(),
                            errorStyle);
                }
            }
        });

        // Add a "Log Out" button
        logoutButton = agentInteractionGui.logoutButton;
        logoutButton.setAction(new AbstractAction("Log Out") {
            public void actionPerformed(ActionEvent actionEvent) {
                try {
                	if(voice)
                		samplePlace.logout(agentInteractionData.getQueue(), null, null);
                	else if(mediaList != null)
                		samplePlace.logoutMultimedia(mediaList, null, null);

                } catch (Exception exception) {
                    exception.printStackTrace();
                    agentInteractionGui.writeLogMessage(exception.getMessage()
                    		+ "cause: " + exception.getCause(),
                            errorStyle);
                }
            }
        });

        // Add a "Ready" button
        readyButton = agentInteractionGui.readyButton;
        readyButton.setAction(new AbstractAction("Ready") {
            public void actionPerformed(ActionEvent actionEvent) {
                try {
                	if(voice)
                	samplePlace.ready(agentInteractionData.getQueue(), null, null, null);
                	else if(mediaList != null)
                	samplePlace.readyMultimedia(mediaList, null, null);
                } catch (Exception exception) {
                    exception.printStackTrace();
                    agentInteractionGui.writeLogMessage(exception.getMessage()
                    		+ "cause: " + exception.getCause(),
                            errorStyle);
                }
            }
        });

        // Add a "Not Ready" button
        notReadyButton = agentInteractionGui.notReadyButton;
        notReadyButton.setAction(new AbstractAction("Not Ready") {
            public void actionPerformed(ActionEvent actionEvent) {
                try {
                	if(voice)
                		samplePlace.notReady(agentInteractionData.getQueue(), null,
                            null, null);
                	else if(mediaList != null)
                		samplePlace.notReadyMultimedia(mediaList, null, null);
                } catch (Exception exception) {
                    exception.printStackTrace();
                    agentInteractionGui.writeLogMessage(exception.getMessage()
                    		+ "cause: " + exception.getCause(),
                            errorStyle);
                }
            }
        });

        // Add an After Call Work ("ACW") button
        afterCallWorkButton = agentInteractionGui.afterCallWorkButton;
        afterCallWorkButton.setAction(new AbstractAction("ACW") {
            public void actionPerformed(ActionEvent actionEvent) {
                try {
                	samplePlace.afterCallWork(agentInteractionData.getQueue(),
                            null, null);
                } catch (Exception exception) {
                    exception.printStackTrace();
                    agentInteractionGui.writeLogMessage(exception.getMessage()
                    		+ "cause: " + exception.getCause(),
                            errorStyle);
                }
            }
        });

        // Add other information
        loginNameLabel = agentInteractionGui.loginNameLabel;
        loginNameLabel.setText("Login Name: "
                + agentInteractionData.getAgent1UserName());
        dnLabel = agentInteractionGui.dnLabel;
        dnLabel.setText("DN: " + agentInteractionData.getDn1());

        agentPlaceLabel = agentInteractionGui.agentPlaceLabel;
        agentPlaceLabel.setText("Place: " + agentInteractionData.getPlace1());
        agentQueueLabel = agentInteractionGui.agentQueueLabel;
        agentQueueLabel.setText("Queue: " + agentInteractionData.getQueue());

        loginButton.setEnabled(true);

        try {
            // THIS IS AN IMPORTANT STEP:
            // Register this application for events on the example place
            samplePlace.addPlaceListener(this);


        } catch (Exception exception) {
            agentInteractionGui.writeLogMessage(exception.getMessage()
            		+ "cause: " + exception.getCause(),
                    errorStyle);
            exception.printStackTrace();
        }
        loginButton.setEnabled(true);
    }
    
    /*public void waitForGuiReady()
    {
    	while(!guiReady)
			try {
				Thread.sleep(500);
			} catch (InterruptedException e) {
				
				e.printStackTrace();
			}
    }*/

    /**
     * Any time the place associated with this example changes its state, the
     * example's widgets must be updated to reflect that state. For example,
     * if the "Log In" button has been clicked and the login was successful,
     * the "Log In" button must now be disabled and the "Log Out" button must
     * be enabled. <br/><br/>
     *
     * This method enables or disables the specified widgets, which are normally
     * just buttons and radio buttons, but may include other widgets. To do
     * this, it uses the isPossible method of the Possible interface. <br/><br/>
     *
     * The isPossible method returns either true or false, depending on
     * whether a particular action is possible. This boolean value is used to
     * determine whether the widget will be enabled.
     */
    public void setPlaceWidgetState() {
    	
    	if(sampleDn != null)
    	{    		
    		loginButton.setEnabled(sampleDn.isPossible(Dn.Action.LOGIN));
    		logoutButton.setEnabled(sampleDn.isPossible(Dn.Action.LOGOUT));
    		readyButton.setEnabled(sampleDn.isPossible(Dn.Action.READY));
    		notReadyButton.setEnabled(sampleDn.isPossible(Dn.Action.NOTREADY));
    		afterCallWorkButton.setEnabled(sampleDn.isPossible(Dn.Action.AFTERCALLWORK));

    		//Set status in the table.
    		agentInteractionGui.setPlaceStatus(sampleDn.getName(),sampleDn.getStatus().toString());
    	}
    } 
   
    /**
     * The setInteractionWidgetState() method is used by subclasses of the
     * SimplePlace class. These classes have widgets whose state must be
     * set when they receive InteractionEvents. <br/><br/>
     *
     * Since SimplePlace itself does not receive InteractionEvents, the method
     * body has been left empty here.
     */
    public void setInteractionWidgetState() {
    }

    /**
     * Receives DnEvents, which report changes to the status or data of a DN.
     * If you implement the PlaceListener interface, you must implement this
     * method.<br/><br/>
     *
     * The SimplePlace example uses this method to log these DnEvents as they
     * occur and to set the state of the widgets used by the SimplePlace example.
     */
    public void handleDnEvent(DnEvent event) {
    	DnEventThread p = new DnEventThread(event);
		p.start();

    }

    /**
     * Receives PlaceEvents, which report changes to the status or data of a DN.
     * If you implement the PlaceListener interface, you must implement this
     * method.<br/><br/>
     *
     * The SimplePlace example uses this method to log these PlaceEvents as they
     * occur.
     */
    public void handlePlaceEvent(PlaceEvent event) {
    	PlaceEventThread p = new PlaceEventThread(event,this);
		p.start();
    }


    /**
     * This method is required by the PlaceListener interface, but is not used
     * by the SimplePlace example because this example doesn't generate any
     * interactions. However, this method is implemented by the examples that
     * use interactions, such as SimpleVoiceInteraction and SimpleEmailInteraction.
     */
    public void handleInteractionEvent(InteractionEvent event) {
    	InteractionEventThread p = new InteractionEventThread(event);
		p.start();
    }

    /**
     * This method is required by the PlaceListener interface, but is not used by
     * the Agent Interaction (Java) code examples.
     */
    public void contactChanged(InteractionEvent event) {
    }

    /**
     * This method is required by the PlaceListener interface, but is not used by
     * the Agent Interaction (Java) code examples.
     */
    public void deleted() {
    }

    /**
     * This method is required by the PlaceListener interface, but is not used
     * by the Agent Interaction (Java) code examples.
     */
    public void dnAdded(String dn) {
    }

    /**
     * This method is required by the PlaceListener interface, but is not used
     * by the Agent Interaction (Java) code examples.
     */
    public void dnRemoved(String dn) {
    }

    /**
     * This method is required by the PlaceListener interface, but is not used
     * by the Agent Interaction (Java) code examples.
     */
    public void idModified(String id) {
    }

    /**
     * This method is used to run the SimplePlace example.
     * @param args There are no arguments for this class.
     */
    public static void main(String[] args) {
        //Schedule a job for the event-dispatching thread:
        //creating and showing this application's GUI.
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                SimplePlace application = new SimplePlace(
                        "Agent Interaction (Java): SimplePlace");
            }
        });
    }
    
    
    
    /**
	 * Creates and submits an interaction
	 * containing the fileToSend file as an attached data.
	 */
	class PlaceEventThread extends Thread {

		PlaceEvent event;
		
		PlaceEventThread(PlaceEvent _event, SimplePlace _simplePlace) {
            event= _event;
        }


        public void run() {
        	
//        	 Get information about the event
            Place place = event.getPlace();
            String placeString = place.getId();
            String eventInfo = "";
            String eventSource = event.getSource().toString();

            if( event instanceof PlaceEventMediaStatusChanged) {

                PlaceEventMediaStatusChanged evt = (PlaceEventMediaStatusChanged) event;

                //Managing the event information
                eventInfo =   "    Status change in " + evt.getMedia().getName()
    						+ "    New status " +evt.getStatus().toString();
                if (agentInteractionGui.createDetailedPlaceEventMessages)
                {
                	eventInfo += "    Reason "+ evt.getReason().toString();
                }

            }
            if (event instanceof PlaceEventMediaAdded)
            {
                PlaceEventMediaAdded evt = (PlaceEventMediaAdded) event;
                //Managing the event information
                String name = evt.getMedia().getName() ;
                if(name.equals("email"))
                	sampleEmail = evt.getMedia();
                else if(name.equals("chat"))
                	sampleChat =  evt.getMedia();

                else
                	sampleOpenMedia =  evt.getMedia();

                eventInfo = "    Added " + name;
            }
            if (event instanceof PlaceEventMediaRemoved)
            {
            	PlaceEventMediaRemoved evt = (PlaceEventMediaRemoved) event;
                //Managing the event information
            	String name = evt.getMedia().getName() ;
                eventInfo = "    Removed " + name ;
                if(name.equals("email"))
                	sampleEmail = null;
                else if(name.equals("chat"))
                	sampleChat =  null;
                else
                	sampleOpenMedia = null;

            }
            // If the user asked for log information, collect it and write it to the
            // application log window
            if (agentInteractionGui.logPlaceEvents) {
                // Set up a string for writing to the application log window
                String placeEventString;

                if (agentInteractionGui.createDetailedPlaceEventMessages) {
                    // If the user asked for detailed log information, collect it
                    placeEventString = "Place " + placeString
    						+ "    Event "+ eventInfo
    						+ "    Medias: " +  event.getPlace().getMedia().toString()
    						+ "    Interactions: " + event.getPlace().getInteractions().toString();
                    // Write the message to the application log window
                    agentInteractionGui.writeLogMessage(placeEventString,
                            placeEventStyle);
                } else {
                    // If the user asked for standard log information, collect it
                    placeEventString = " Place: " + placeString +  "    Event "+ eventInfo ;
                    // Write the message to the application log window
                    agentInteractionGui.writeLogMessage(placeEventString,
                            placeEventStyle);
                }

            }

            // THIS IS AN IMPORTANT STEP:
            // As the status changes, enable or disable the buttons
            setPlaceWidgetState();

            //Changes can affect some place widgets used for interaction in further examples
            setInteractionWidgetState();
            
     

            
        }
	}
	
	class DnEventThread extends Thread {

		DnEvent event;
		
		public DnEventThread(DnEvent _event) {
            event= _event;
        }

		
		public void run()
		{
		     // Get information about the event
	        Dn dn = event.getDn();
	        String dnString = dn.getId();
	        String dnStatusString = dn.getStatus().toString();
	        String eventReason = event.getEventReason().toString();

	        // If the user asked for log information, collect it and write it to the
	        // application log window
	        if (agentInteractionGui.logDnEvents) {

	            // Set up a string for writing to the application log window
	            String dnEventString;

	            if (agentInteractionGui.createDetailedDnEventMessages) {

	            	// If the user asked for detailed log information, collect it
	                dnEventString = "DN: " + dnString + "   "
	                        + dnStatusString + " (" + event.getSource() + ")";

	                // Write the message to the application log window
	                agentInteractionGui.writeLogMessage(dnEventString, dnEventStyle);

	            } else {
	                // If the user asked for standard log information, collect it
	                dnEventString = "DN: "   + dnString
	                        + "    Status: " + dnStatusString
							+ "    Reason: " + eventReason ;
	                // Write the message to the application log window
	                agentInteractionGui.writeLogMessage(dnEventString, dnEventStyle);
	            }
	        }

	        // THIS IS AN IMPORTANT STEP:
	        // As the status changes, enable or disable the buttons
	       	setPlaceWidgetState();
	      	//A change of status in a DN could make some actions no longer possible
	       	//on interactions of the inherited examples (MakeCall)
	       	setInteractionWidgetState();
		}
	}
	
	class InteractionEventThread extends Thread
	{
		InteractionEvent event;
		
		public InteractionEventThread(InteractionEvent _event) {
            event= _event;
        }
		
		public void run()
		{
			// If the user asked for log information, collect it and write it to the
	        // application log window
	    	// Any event is displayed, regardless the concerned interaction type.
	    	try{
	    	Interaction interaction = event.getInteraction();
	        if (agentInteractionGui.logInteractionEvents) {
	            // Set up a string for writing to the application log window
	            String interactionEventString;

	            if (agentInteractionGui.createDetailedInteractionEventMessages) {
	                // If the user asked for detailed log information, collect it
	                interactionEventString = "Interaction Type "
					+ event.getInteraction().getType().toString()
					+ "    ID "+event.getInteraction().getId()
					+ "    Status: "+event.getStatus().toString()
					+ "    Status changed: "+event.isStatusChanged()
	                + "    Reason: " + event.getEventReason()
	                + "    Done: "+interaction.isDone()
					+ "    Dirty: "+interaction.isDirty()
					+ "    Queue:  "+interaction.getQueue()
					+ "    Subject:  "+ interaction.getSubject()
					+ "    Extensions:" + event.getExtensions().toString();

	                // Write the message to the application log window
	                agentInteractionGui.writeLogMessage(interactionEventString,
	                        interactionEventStyle);
	            } else {
	                // If the user asked for standard log information, collect it
	                interactionEventString = "Interaction  Type "
					+ event.getInteraction().getType().toString()
					+ " ID "+event.getInteraction().getId()
					+ "    Status: "+event.getStatus().toString()
					+ "    Status changed: "+event.isStatusChanged()
					+ "    Reason: " + event.getEventReason() ;

	                // Write the message to the application log window
	                agentInteractionGui.writeLogMessage(interactionEventString,
	                        interactionEventStyle);
	            }
	        }
	    	}catch(Exception __e)
			{
	    		__e.printStackTrace();
			}
		}
	}

}