/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2007-2008 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
package agent.interaction.samples;

import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;


import com.genesyslab.ail.Dn;
import com.genesyslab.ail.Interaction;
import com.genesyslab.ail.InteractionVoice;
import com.genesyslab.ail.MediaType;
import com.genesyslab.ail.event.InteractionEvent;
import com.genesyslab.ail.exception.RequestFailedException;

/**
 * This example uses a GUI application to show how to make and answer a call,
 * place the call on hold, release the call, and mark it done. <br/><br/>
 *
 * One PlaceListener is registered, which receives InteractionEvents.<br/><br/>
 */
public class SimpleVoiceInteraction extends SimplePlace {

	InteractionVoice sampleInteraction;

	JButton answerButton;
	JButton doneButton;
	JButton releaseButton;
	JButton makeCallButton;
	JButton holdButton;
	JButton retrieveButton;

	JLabel simpleVoiceTargetDnText;
	JTextField simpleVoiceTargetDn;
	JLabel interactionDnis;
	JLabel interactionAni;
	JLabel interactionSubject;
	JLabel interactionParties;

	/**
	 * Creates an instance of SimpleVoiceInteraction.<br/>
	 *
	 * The super constructor connects to the Agent Interaction Layer.
	 *
	 * After connecting, it sets up an agent, place, media, and DN for use by
	 * the examples. Then it creates the GUI, links the GUI components (widgets)
	 * to API functionality, and starts the application.
	 *
	 * @param windowTitle
	 *            The title of the GUI window for this example.
	 */
	public SimpleVoiceInteraction(String windowTitle) {
		super(windowTitle);
	}

	/**
	 * Tells the AgentInteractionGui class which example has called
	 * the GUI, so that it can determine which widgets will be enabled when the
	 * GUI is created.
	 */
	public void setSampleType() {
		// Let the GUI class know what type of example this is
		sampleType = "SimpleVoiceInteraction";
	}

	/**
	 * Links widgets (user interface
	 * components) to API functionality and registers a listener for
	 * events on the Place the agent has logged in to.
	 * <br/><br/>
	 * The linked widgets for Simple Voice are buttons for processing a voice interaction.
	 * These buttons correspond to the following actions:<ul>
	 * <li>Answer</li>
	 * <li>Make call</li>
	 * <li>Release</li>
	 * <li>Hold and retrieve</li>
	 * <li>Mark done</li>
	 * </ul>
	 * @see SimplePlace#linkWidgetsToGui()
	 */
	public void linkWidgetsToGui() {
		// Execute the corresponding method in SimplePlace first
		super.linkWidgetsToGui();


		// Add an Answer button
		// For each button, the first step is to link that button to the
		// corresponding GUI component...
		answerButton = agentInteractionGui.answerButton;
		// Then you can set an action for the button
		answerButton.setAction(new AbstractAction("Answer") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Answer an incoming call
					sampleInteraction.answerCall(null);
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Add a Make Call button
		makeCallButton = agentInteractionGui.makeCallButton;
		makeCallButton.setAction(new AbstractAction("Make Call") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Create a new interaction for use in making the call
					sampleInteraction =
						(InteractionVoice) samplePlace.createInteraction(MediaType.VOICE, null,
								agentInteractionData.getQueue());
					if (sampleInteraction instanceof InteractionVoice) {
						// Make the call, using the phone number provided by
						// the agent
						sampleInteraction.makeCall(
								simpleVoiceTargetDn.getText(), null,
								InteractionVoice.MakeCallType.REGULAR, null,
								null, null);
					}
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Add a Release Call button
		releaseButton = agentInteractionGui.releaseButton;
		releaseButton.setAction(new AbstractAction("Release") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Release the call
					sampleInteraction.releaseCall(null);
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Add a Mark Done button
		doneButton = agentInteractionGui.doneButton;
		doneButton.setAction(new AbstractAction("Done") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Mark the call as done
					sampleInteraction.markDone();
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Add a Hold button
		holdButton = agentInteractionGui.holdButton;
		holdButton.setAction(new AbstractAction("Hold") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Hold the call
					sampleInteraction.holdCall(null, null);
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Add a Retrieve button
		retrieveButton = agentInteractionGui.retrieveButton;
		retrieveButton.setAction(new AbstractAction("Retrieve") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Retrieve the held call
					sampleInteraction.retrieveCall(null,null);
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Link these fields to the corresponding GUI components
		simpleVoiceTargetDnText = agentInteractionGui.simpleVoiceTargetDnLabel;
		simpleVoiceTargetDn = agentInteractionGui.simpleVoiceTargetDnTextField;

		interactionDnis = agentInteractionGui.interactionDnisLabel;
		interactionAni = agentInteractionGui.interactionAniLabel;
		interactionSubject = agentInteractionGui.interactionSubjectLabel;
		interactionParties = agentInteractionGui.interactionPartiesLabel;

	}

	/**
	 * This class receives InteractionEvents, which are handled by the
	 * handleInteractionEvent() method. This class also has widgets whose state
	 * must be set when it receives these events.
	 *
	 * This method enables or disables the specified widgets, which are normally
	 * just buttons and radio buttons, but may include other widgets. To do
	 * this, it uses the isPossible() method of the Possible interface.
	 *
	 * The isPossible() method returns either true or false, depending on
	 * whether a particular action is possible. This boolean value is used to
	 * determine whether the widget will be enabled.
	 */
	public void setInteractionWidgetState() {

		if (sampleInteraction !=null) {
			answerButton.setEnabled(sampleInteraction
					.isPossible(InteractionVoice.Action.ANSWER_CALL));
			makeCallButton.setEnabled(sampleInteraction
					.isPossible(InteractionVoice.Action.MAKE_CALL));
			holdButton.setEnabled(sampleInteraction
					.isPossible(InteractionVoice.Action.HOLD));
			retrieveButton.setEnabled(sampleInteraction
					.isPossible(InteractionVoice.Action.RETRIEVE));
			releaseButton.setEnabled(sampleInteraction
					.isPossible(InteractionVoice.Action.RELEASE_CALL));
			doneButton.setEnabled(sampleInteraction
					.isPossible(InteractionVoice.Action.MARK_DONE));

			interactionDnis.setText("Dnis:"+sampleInteraction.getDNIS());
			interactionAni.setText("Ani: "+sampleInteraction.getANI());
			interactionSubject.setText("Subject:"+sampleInteraction.getSubject());
			interactionParties.setText("Parties: "+sampleInteraction.getParties().toString());
		}else{
			answerButton.setEnabled(false);
			makeCallButton.setEnabled(sampleDn.isPossible(Dn.Action.MAKE_CALL));
			releaseButton.setEnabled(false);
			doneButton.setEnabled(false);
			holdButton.setEnabled(false);
			retrieveButton.setEnabled(false);

			interactionDnis.setText("Dnis:");
			interactionAni.setText("Ani: ");
			interactionSubject.setText("Subject:");
			interactionParties.setText("Parties: ");
		}
	}

	/**
	 * Receives InteractionEvents, which report changes to the status or data
	 * of an Interaction of this place. If you implement the PlaceListener interface, you must
	 * implement this method. <br/><br/>
	 *
	 * The SimpleVoiceInteraction code example uses this method to log these
	 * InteractionEvents as they occur and, if the event involves a voice interaction
	 * to set the state of the widgets
	 * used by the SimpleVoiceInteraction and MultipartyVoiceInteraction
	 * examples.
	 */
	public void handleInteractionEvent(InteractionEvent event) {

		//Display the event by calling the super SimplePlace.handleInteractionEvent()
		super.handleInteractionEvent(event);
		VoiceInteractionEventThread p = new VoiceInteractionEventThread(event);
		p.start();	

	
	}
	/**
	 * This method is used to run the SimpleVoiceInteraction example.
	 * @param args There are no arguments for this class.
	 */
	public static void main(String[] args) {
		//Schedule a job for the event-dispatching thread:
		//creating and showing this application's GUI.
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				SimpleVoiceInteraction application = new SimpleVoiceInteraction(
				"Agent Interaction (Java): Simple Voice Interaction");
			}
		});
	}
	/**
	 * 
	 * @author syvenou
	 *
	 * TODO To change the template for this generated type comment go to
	 * Window - Preferences - Java - Code Style - Code Templates
	 */
	class VoiceInteractionEventThread extends Thread
	{
		InteractionEvent event;
		
		public VoiceInteractionEventThread(InteractionEvent _event)
		{
			event=_event;
			
		}
		
		public void run()
		{
			try {

				if(event.getSource() instanceof InteractionVoice)
				{
					// Get information about the event
					InteractionVoice interaction = (InteractionVoice) event.getInteraction();

					//if sampleInteraction is null, this event may indicate a ringing interaction
					// In this example, we handle only the RINGING case
					if (sampleInteraction == null
							&& event.getStatus() == Interaction.Status.RINGING) {
						// Associate sampleInteraction with the event source
						sampleInteraction = (InteractionVoice) event.getSource();
						simpleVoiceTargetDn.setText(sampleInteraction.getParties().toString());
					}

					// If the interaction is idle and done, the example no longer handles it.
					if (sampleInteraction!=null && interaction.getId() == sampleInteraction.getId()
							&& event.getStatus() == Interaction.Status.IDLE
							&& sampleInteraction.isDone() )
					{
						sampleInteraction = null;
						simpleVoiceTargetDn.setText("");
					}

					//Update the widgets
					setInteractionWidgetState();
				}
		}	catch (RequestFailedException e) {
			e.printStackTrace();
		}
		}
	}
}