/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2007-2008 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
package agent.interaction.samples.outbound;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.apache.log4j.Logger;
import agent.interaction.samples.AgentInteractionData;

import com.genesyslab.ail.Contact;
import com.genesyslab.ail.Dn;
import com.genesyslab.ail.Interaction;
import com.genesyslab.ail.InteractionVoice;
import com.genesyslab.ail.MediaType;
import com.genesyslab.ail.OutboundChain;
import com.genesyslab.ail.OutboundRecord;
import com.genesyslab.ail.OutboundService;
import com.genesyslab.ail.Place;
import com.genesyslab.ail.event.PlaceEventOutboundChainInfo;
import com.genesyslab.ail.event.InteractionEvent;
import com.genesyslab.ail.event.PlaceEvent;
import com.genesyslab.ail.exception.RequestFailedException;

/**
 * Manages voice, record, and chain widgets for the selected interaction.
 * <br>At runtime, an instance of this class lists the voice interactions of the monitored place in an interaction table. If the user selects one of then, voice, record, and chain panels update.
 *  this enables the user to process the outbound interaction, including records and chain data.

 */
public class VoiceInteractionAdapterForOutbound  {

	/** Adds logs labeled "ForOutbound.VoiceInteractionHandler" in the log file */ 
	Logger sLog = Logger.getLogger("ForOutbound.VoiceInteractionAdapter");

	InteractionVoice selectedInteraction;
	AgentOutboundGui outboundGui;
	Place samplePlace;
	Dn sampleDn;
	AgentInteractionData agentInteractionData;
	HashMap<String, InteractionVoice> interactionIds = new HashMap<String, InteractionVoice>();
	OutboundService outboundService;

	JTable ixnTable;
	JButton answerButton;
	JButton doneButton;
	JButton releaseButton;
	JButton makeCallButton;
	JButton holdButton;
	JButton retrieveButton;

	JLabel simpleVoiceTargetDnText;
	JTextField simpleVoiceTargetDn;
	JLabel interactionDnis;
	JLabel interactionAni;
	JLabel interactionSubject;
	JLabel interactionParties;

	//current record widgets
	OutboundRecord selectedRecord;
	JComboBox callResultComboBox;
	JComboBox phoneTypeComboBox;
	JLabel phoneLabel,dailyFromLabel,dailyTillLabel;
	JLabel phoneTypeLabel;

	JButton cancelCurrentRecordButton;
	JButton doNotCallCurrentRecordButton;
	JButton rescheduleCurrentRecordButton;
	JButton updateFromOCSCurrentRecordButton;
	JLabel currentRecordStatusLabel;

	// current chain widgets
	OutboundChain selectedChain;
	JButton cancelChainButton;
	JButton doNotCallChainButton;
	JButton closeChainButton;
	JButton rejectChainButton;
	JLabel chainIsScheduledLabel;
	JButton requestRecordChainButton;
	JButton markProcessedChainButton;
	JComboBox recordsComboBox;
	JComboBox chainTreatmentComboBox;
	ComboBoxListener cbbListener;

	//current contact widgets
	Contact selectedContact;

	/**
	 * Creates an instance of VoiceInteractionHandlerForOutbound.<br/>
	 *
	 */
	public VoiceInteractionAdapterForOutbound(Place _samplePlace, Dn _sampleDn, OutboundService _outboundService, AgentInteractionData _agentInteractionData, AgentOutboundGui _gui) {

		// Set up an agent, place and DN for use by the examples
		samplePlace = _samplePlace;
		sampleDn = _sampleDn;
		outboundGui = _gui;
		agentInteractionData = _agentInteractionData;
		outboundService = _outboundService;

		linkWidgetsToGui();
		sLog.info("Initialisation OK");
	}



	/**
	 * Links widgets (user interface
	 * components) to API functionality.
	 * The linked widgets are buttons for processing:
	 * <ul><li>A voice interaction selected in the interaction table.</li>
	 * <li>An outbound record.</li>
	 * <li>An outbound chain (which chained a selection of records).</li></ul>
	 * </ul>
	 * @see SimplePlace#linkWidgetsToGui()
	 */
	public void linkWidgetsToGui() {

		//////////////// VOICE CONTROLS /////////////////////
		// Add an Answer button
		// For each button, the first step is to link that button to the
		// corresponding GUI component...
		answerButton = outboundGui.answerButton;
		// Then you can set an action for the button
		answerButton.setAction(new AbstractAction("Answer") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Answer an incoming call
					selectedInteraction.answerCall(null);
				} catch (Exception exception) {
					sLog.error(" answerButton "+ exception.getLocalizedMessage());
				}
			}
		});
		answerButton.setEnabled(false);

		// Add a Make Call button
		makeCallButton = outboundGui.makeCallButton;
		makeCallButton.setAction(new AbstractAction("Make Call") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Create a new interaction for use in making the call
					selectedInteraction =
						(InteractionVoice) samplePlace.createInteraction(MediaType.VOICE, null,
								agentInteractionData.getQueue());
					if (selectedInteraction instanceof InteractionVoice) {
						// Make the call, using the phone number provided by
						// the agent
						selectedInteraction.makeCall(
								simpleVoiceTargetDn.getText(), null,
								InteractionVoice.MakeCallType.REGULAR, null,
								null, null);
					}
				} catch (Exception exception) {
					sLog.error(" makeCallButton "+ exception.getLocalizedMessage());
				}
			}
		});
		makeCallButton.setEnabled(false);

		// Add a Release Call button
		releaseButton = outboundGui.releaseButton;
		releaseButton.setAction(new AbstractAction("Release") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Release the call
					selectedInteraction.releaseCall(null);
				} catch (Exception exception) {
					sLog.error(" releaseButton "+ exception.getLocalizedMessage());
				}
			}
		});
		releaseButton.setEnabled(false);

		// Add a Mark Done button
		doneButton = outboundGui.doneButton;
		doneButton.setAction(new AbstractAction("Done") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Mark the call as done
					selectedInteraction.markDone();
				} catch (Exception exception) {
					sLog.error(" doneButton "+ exception.getLocalizedMessage());
				}
			}
		});
		doneButton.setEnabled(false);

		// Add a Hold button
		holdButton = outboundGui.holdButton;
		holdButton.setAction(new AbstractAction("Hold") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Hold the call
					selectedInteraction.holdCall(null, null);
				} catch (Exception exception) {
					sLog.error(" holdButton "+ exception.getLocalizedMessage());
				}
			}
		});
		holdButton.setEnabled(false);

		// Add a Retrieve button
		retrieveButton = outboundGui.retrieveButton;
		retrieveButton.setAction(new AbstractAction("Retrieve") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Retrieve the held call
					selectedInteraction.retrieveCall(null,null);
				} catch (Exception exception) {
					sLog.error(" retrieveButton "+ exception.getLocalizedMessage());
				}
			}
		});
		retrieveButton.setEnabled(false);

		// Link these fields to the corresponding GUI components
		simpleVoiceTargetDnText = outboundGui.simpleVoiceTargetDnLabel;
		simpleVoiceTargetDn = outboundGui.simpleVoiceTargetDnTextField;

		interactionDnis = outboundGui.interactionDnisLabel;
		interactionAni = outboundGui.interactionAniLabel;
		interactionSubject = outboundGui.interactionSubjectLabel;
		interactionParties = outboundGui.interactionPartiesLabel;


		///////////// INTERACTION TABLE /////////////////////
		
		ixnTable = outboundGui.ixnTable;
		InteractionSelectionListener listener = new InteractionSelectionListener(ixnTable);
		ixnTable.getSelectionModel().addListSelectionListener(listener);
		
		
		///////////// RECORDS CONTROLS ////////////////////////
		
		JTable chainedRecordsTable = outboundGui.chainedRecordsTable;
		ChainedRecordSelectionListener chlistener = new ChainedRecordSelectionListener(chainedRecordsTable);
		chainedRecordsTable.getSelectionModel().addListSelectionListener(chlistener);


		cancelCurrentRecordButton= outboundGui.cancelCurrentRecordButton;
		cancelCurrentRecordButton.setAction(new AbstractAction("Cancel") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					selectedRecord.cancel();					
				} catch (Exception exception) {
					sLog.error(" cancelCurrentRecordButton "+ exception.getLocalizedMessage());
				}
			}
		});
		cancelCurrentRecordButton.setEnabled(false);

		doNotCallCurrentRecordButton= outboundGui.doNotCallCurrentRecordButton;
		doNotCallCurrentRecordButton.setAction(new AbstractAction("Do not call") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					selectedRecord.doNotCall("No message");
				} catch (Exception exception) {
					sLog.error(" doNotCallCurrentRecordButton "+ exception.getLocalizedMessage());
				}
			}
		});
		doNotCallCurrentRecordButton.setEnabled(false);

		rescheduleCurrentRecordButton= outboundGui.rescheduleCurrentRecordButton;
		rescheduleCurrentRecordButton.setEnabled(false);



		currentRecordStatusLabel= outboundGui.currentRecordStatusLabel;
		phoneLabel= outboundGui.phoneLabel;
		dailyFromLabel= outboundGui.dailyFromLabel;
		dailyTillLabel= outboundGui.dailyTillLabel;
		phoneTypeLabel= outboundGui.phoneTypeLabel;


		callResultComboBox = outboundGui.callResultComboBox;
		for(int i=0;i<OutboundRecord.CallResult.max();i++)
		{
			OutboundRecord.CallResult v = OutboundRecord.CallResult.getCallResult(i);
			if (v!=null)
			{
				String val =  v.toString();
				callResultComboBox.addItem(val);
			}
		}    	

		callResultComboBox.setEnabled(false);

		phoneTypeComboBox= outboundGui.phoneTypeComboBox;
		for(int i=0;i<OutboundRecord.PhoneType.max();i++)
		{
			OutboundRecord.PhoneType v = OutboundRecord.PhoneType.getPhoneType(i);
			if (v!=null)
			{
				String val =  v.toString();    		
				phoneTypeComboBox.addItem(val);
			}
		}
		phoneTypeComboBox.setEnabled(false);	
		
		
		/////////////// CHAIN WIDGETS //////////////////

		cancelChainButton = outboundGui.cancelChainButton;
		cancelChainButton.setAction(new AbstractAction("Cancel") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					selectedChain.cancel();
				} catch (Exception exception) {
					sLog.error(" cancelChainButton "+ exception.getLocalizedMessage());
				}
			}
		});
		cancelChainButton.setEnabled(false);


		doNotCallChainButton = outboundGui.doNotCallChainButton;
		doNotCallChainButton.setAction(new AbstractAction("Do not call") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					selectedChain.doNotCall("No message");
				} catch (Exception exception) {
					sLog.error(" doNotCallChainButton "+ exception.getLocalizedMessage());
				}
			}
		});
		doNotCallChainButton.setEnabled(false);

		closeChainButton = outboundGui.closeChainButton;
		closeChainButton.setAction(new AbstractAction("Close chain") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					selectedChain.close();
				} catch (Exception exception) {
					sLog.error(" closeChainButton "+ exception.getLocalizedMessage());
				}
			}
		});
		closeChainButton.setEnabled(false);


		rejectChainButton = outboundGui.rejectChainButton;
		rejectChainButton.setAction(new AbstractAction("Reject chain") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					selectedChain.reject();
				} catch (Exception exception) {
					sLog.error(" rejectChainButton "+ exception.getLocalizedMessage());
				}
			}
		});
		rejectChainButton.setEnabled(false);

		chainIsScheduledLabel  = outboundGui.chainIsScheduledLabel;

		requestRecordChainButton= outboundGui.requestRecordChainButton;
		requestRecordChainButton.setAction(new AbstractAction("Request records") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					selectedChain.requestChainedRecords();
				} catch (Exception exception) {
					sLog.error(" requestChainedRecords "+ exception.getLocalizedMessage());
				}
			}
		});
		requestRecordChainButton.setEnabled(false);


		markProcessedChainButton= outboundGui.markProcessedChainButton;
		markProcessedChainButton.setAction(new AbstractAction("Mark processed") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					selectedChain.markProcessed();
				} catch (Exception exception) {
					sLog.error(" markProcessedChainButton "+ exception.getLocalizedMessage());
				}
			}
		});
		markProcessedChainButton.setEnabled(false);


		chainTreatmentComboBox = outboundGui.chainTreatmentComboBox;
		for(int i=0;i<OutboundChain.TreatmentType.max();i++)
		{
			OutboundChain.TreatmentType v = OutboundChain.TreatmentType.getTreatmentType(i);
			if (v!=null)
			{
				String val =  v.toString();

				chainTreatmentComboBox.addItem(val);
			}
		}
		chainTreatmentComboBox.addActionListener(cbbListener);
		chainTreatmentComboBox.setEnabled(false);
		cbbListener = new ComboBoxListener();
		sLog.debug(" Widgets linked to GUI OK");
	}


	/**
	 * This method enables or disables the specified widgets, which are normally
	 * just buttons and radio buttons, but may include other widgets. To do
	 * this, it uses the isPossible() method of the Possible interface.
	 *
	 * The isPossible() method returns either true or false, depending on
	 * whether a particular action is possible. This boolean value is used to
	 * determine whether the widget will be enabled.
	 */
	public synchronized void setInteractionWidgetState() {

		if (selectedInteraction !=null) {

			simpleVoiceTargetDn.setText(selectedInteraction.getParties().toString());

			answerButton.setEnabled(selectedInteraction
					.isPossible(InteractionVoice.Action.ANSWER_CALL));
			makeCallButton.setEnabled(selectedInteraction
					.isPossible(InteractionVoice.Action.MAKE_CALL));
			holdButton.setEnabled(selectedInteraction
					.isPossible(InteractionVoice.Action.HOLD));
			retrieveButton.setEnabled(selectedInteraction
					.isPossible(InteractionVoice.Action.RETRIEVE));
			releaseButton.setEnabled(selectedInteraction
					.isPossible(InteractionVoice.Action.RELEASE_CALL));
			doneButton.setEnabled(selectedInteraction
					.isPossible(InteractionVoice.Action.MARK_DONE));

			interactionDnis.setText("Dnis:"+selectedInteraction.getDNIS());
			interactionAni.setText("Ani: "+selectedInteraction.getANI());
			interactionSubject.setText("Subject:"+selectedInteraction.getSubject());
			interactionParties.setText("Parties: "+selectedInteraction.getParties().toString());

			/*** update the interaction property tree */
			outboundGui.switchPropertyTree("Interaction properties",selectedInteraction.getAttachedData());
			sLog.debug("Updated widgets for interaction "+selectedInteraction.getId());

		}else{
			answerButton.setEnabled(false);
			makeCallButton.setEnabled(sampleDn.isPossible(Dn.Action.MAKE_CALL));
			releaseButton.setEnabled(false);
			doneButton.setEnabled(false);
			holdButton.setEnabled(false);
			retrieveButton.setEnabled(false);

			interactionDnis.setText("Dnis:");
			interactionAni.setText("Ani: ");
			interactionSubject.setText("Subject:");
			interactionParties.setText("Parties: ");
			outboundGui.clearProperties("Interaction properties");

			sLog.debug("Updated widgets for null interaction");
		}

	}

	/**
	 * Enables or disables record widgets accordingly with the selected record's data.
	 */
	public synchronized void setRecordWidgets()
	{
		if(selectedRecord == null)
		{
			callResultComboBox.removeActionListener(cbbListener);
			phoneTypeComboBox.removeActionListener(cbbListener);

			callResultComboBox.setSelectedIndex(0);
			callResultComboBox.setEnabled(false);
			phoneTypeComboBox.setSelectedIndex(0);
			phoneTypeComboBox.setEnabled(false);

			cancelCurrentRecordButton.setEnabled(false);			
			doNotCallCurrentRecordButton.setEnabled(false);
			rescheduleCurrentRecordButton.setEnabled(false);

			currentRecordStatusLabel.setText("No status");
			phoneLabel.setText("--");
			dailyFromLabel.setText("--");
			dailyTillLabel.setText("--");

			outboundGui.clearProperties("Custom fields");

			sLog.debug("Updated widgets for null record");

		}else
		{
			try{
				callResultComboBox.removeActionListener(cbbListener);
				phoneTypeComboBox.removeActionListener(cbbListener);
			}catch(Exception e){}

			sLog.debug(selectedRecord.toString());

			try{
				cancelCurrentRecordButton.setEnabled(selectedRecord.isPossible(OutboundRecord.Action.CANCEL));
				doNotCallCurrentRecordButton.setEnabled(selectedRecord.isPossible(OutboundRecord.Action.DO_NOT_CALL));
				currentRecordStatusLabel.setText(selectedRecord.getStatus().toString());

				outboundGui.switchPropertyTree("Custom properties",selectedRecord.getCustomFields());

				callResultComboBox.setEnabled(true);
				callResultComboBox.setSelectedItem(selectedRecord.getCallResult().toString());			
				sLog.debug("CallResult in GUI is " + callResultComboBox.getSelectedItem().toString());


				phoneTypeComboBox.setEnabled(true);
				phoneTypeComboBox.setSelectedItem(selectedRecord.getPhoneType().toString());
				sLog.debug("Phone Type in GUI is " + selectedRecord.getPhoneType().toString());

				long from_l = 0;
				long till_l = 0;

				from_l= selectedRecord.getDailyFrom()*1000;
				till_l= selectedRecord.getDailyTill()*1000;

				Date from = new Date(from_l);
				Date till = new Date(till_l);
				dailyFromLabel.setText(from.toString().substring(10));
				dailyTillLabel.setText(till.toString().substring(10));


				phoneLabel.setText(selectedRecord.getPhone());


			}catch(Exception __e)
			{
				sLog.error("setRecordWidgets "+ __e.getLocalizedMessage());
			}


			callResultComboBox.addActionListener(cbbListener);
			phoneTypeComboBox.addActionListener(cbbListener);
			sLog.debug("Updated widgets for selected record");

		}

	}
	/**
	 * Enables or disables record widgets accordingly with the current chain's data.
	 */
	public synchronized void setChainWidgets()
	{
		if(selectedChain != null)
		{
			try{
				chainTreatmentComboBox.removeActionListener(cbbListener);
			}catch(Exception e){}
			try
			{ 
				cancelChainButton.setEnabled(true);
				doNotCallChainButton.setEnabled(true);
				closeChainButton.setEnabled(true);
				rejectChainButton.setEnabled(true);
				requestRecordChainButton.setEnabled(true);
				markProcessedChainButton.setEnabled(true);
				chainTreatmentComboBox.setEnabled(true);
				chainIsScheduledLabel.setText((selectedChain.isScheduled())?"Scheduled":"Not scheduled");

				Iterator records = selectedChain.getRecords().iterator();
				while(records.hasNext())
				{
					OutboundRecord record = (OutboundRecord) records.next();
					outboundGui.setChainedRecord(record.getId(), record.getPhone(), record.getStatus().toString(), record.getCallResult().toString());
				}
			}catch(Exception e) {

				sLog.error("setChainWidget " + e.getLocalizedMessage());
			}
			chainTreatmentComboBox.addActionListener(cbbListener);

		}else
		{
			chainTreatmentComboBox.removeActionListener(cbbListener);
			chainTreatmentComboBox.setEnabled(false);
			doNotCallChainButton.setEnabled(false);
			closeChainButton.setEnabled(false);
			cancelChainButton.setEnabled(false);
			rejectChainButton.setEnabled(false);
			chainIsScheduledLabel.setText("Unknown");
			requestRecordChainButton.setEnabled(false);
			markProcessedChainButton.setEnabled(false);
			outboundGui.clearChainedRecords();
		}

	}

	/**
	 * Receives InteractionEvent, which report changes to one interaction of the place. 
	 * Because this method is called in the PlaceListener, 
	 * this method creates a thread to manage the event without delaying the publication of further incoming events. 
 	 */
	public void handleInteractionEvent(InteractionEvent event) {
		sLog.debug(" Interaction event to process "+event);
		//Display the event by calling the super SimplePlace.handleInteractionEvent()
		VoiceInteractionEventThread p = new VoiceInteractionEventThread(event);
		p.start();	
		sLog.debug(" Ixn thread launched ");


	}
	/**
	 * Manages an InteractionEvent to update the interaction table, and if the event is about the selected interaction, 
	 * to update the chain and record panels.
	 */
	class VoiceInteractionEventThread extends Thread
	{
		InteractionEvent event;

		public VoiceInteractionEventThread(InteractionEvent _event)
		{
			event=_event;
		}

		public void run()
		{
			try {

				if(event.getSource() instanceof InteractionVoice)
				{
					// Get information about the event
					InteractionVoice interaction = (InteractionVoice) event.getInteraction();

					String id =interaction.getId();

					if (selectedInteraction!=null 
							&& interaction.getId().equals(selectedInteraction.getId()))
					{
						// If the interaction is idle and done, the example no longer handles it.
						if( event.getStatus() == Interaction.Status.IDLE
								&& interaction.isDone() )
						{

							sLog.debug(" selected ixn " + id + " is IDLE and done");
							outboundGui.removeIxnStatusInTable(id);
							if(interactionIds.containsKey(id))
								interactionIds.remove(id);
							if(interactionIds.size()==0)
								selectedInteraction = null;
							else
							{
								String next_id =  (String) interactionIds.keySet().iterator().next();
								selectedInteraction = (InteractionVoice) interactionIds.get(next_id); 
							}
						}else
						{
							outboundGui.setInteractionStatusInTable(interaction.getId(),interaction.getStatus().toString(), interaction.getType().toString(), interaction.getSubject());
						}
					} else if(selectedInteraction != null && interaction.getId() != selectedInteraction.getId())
					{ 
						if(!interactionIds.containsKey(interaction.getId()))
							interactionIds.put(interaction.getId(), interaction);
						outboundGui.setInteractionStatusInTable(interaction.getId(),interaction.getStatus().toString(), interaction.getType().toString(), interaction.getSubject());							

					}else if(selectedInteraction == null)
					{
						selectedInteraction = interaction;
						interactionIds.put(interaction.getId(), interaction);
						outboundGui.setInteractionStatusInTable(interaction.getId(),interaction.getStatus().toString(), interaction.getType().toString(), interaction.getSubject());							

					}
					sLog.debug(" event ixn " +interaction.getId() + event.toString());


					updateSelectedReferences();


				}

			}catch (RequestFailedException e) {

				sLog.error("VoiceInteractionEventThread "+ e.getLocalizedMessage());
			}

		}
	}
	/**
	 * Receives PlaceEvent, which may report changes to one chain of the place. 
	 * Because this method is called in the PlaceListener, 
	 * this method creates a thread to manage the event without delaying the publication of further incoming events. 
	 */
	public void handlePlaceEvent(PlaceEvent evt)
	{
		PlaceEventThread p = new PlaceEventThread(evt);
		p.start();
		sLog.debug(" Place event thread launched");
	}


	/**
	 * Handles one PlaceEventOutboundChainInfo event to update chain and record widgets.
	 */
	class PlaceEventThread extends Thread
	{
		PlaceEvent event;

		public PlaceEventThread(PlaceEvent event)
		{
			this.event=event;

		}

		public void run()
		{
			if(event instanceof PlaceEventOutboundChainInfo)
			{
				PlaceEventOutboundChainInfo info = (PlaceEventOutboundChainInfo) event;

				if(info.getOutboundChain().equals(selectedChain))
				{
					selectedChain = info.getOutboundChain();
					selectedRecord = selectedChain.getActiveRecord();
					setRecordWidgets();
					setChainWidgets();
					setInteractionWidgetState();
					sLog.debug("PlaceEventThread: outbound event "+info.getReason().toString());
				}

			}

		}

	}

	/**
	 * Retrieves outbound objects and then, updates widgets. 
	 */
	public void updateSelectedReferences()
	{
		try {
			if(selectedInteraction!=null)
			{
				selectedChain = outboundService.getChain(selectedInteraction);
				sLog.info("nb chains is "+outboundService.getChains().size());
				if(selectedChain!=null)
				{
					selectedRecord = selectedChain.getActiveRecord();
					if(selectedRecord != null)
						sLog.debug("Ixn selection - record is "+selectedRecord.toString());
				}
			}else{
				selectedChain = null;
				selectedRecord = null;
			}


		} catch (Exception e1) {
			sLog.error("updateSelectedReferences "+e1.getLocalizedMessage());
		}

		setInteractionWidgetState();
		setRecordWidgets();
		setChainWidgets();
	}

	/**
	 * Handles selection in the interaction table.
	 */
	public class InteractionSelectionListener implements ListSelectionListener {
		JTable table;

		// It is necessary to keep the table since it is not possible
		// to determine the table from the event's source
		public InteractionSelectionListener(JTable table) {
			this.table = table;
		}
		public void valueChanged(ListSelectionEvent e) {
			// If cell selection is enabled, both row and column change events are fired

			ListSelectionModel lsm = (ListSelectionModel)e.getSource();
			if (!lsm.isSelectionEmpty()) {
				int selectedRow = lsm.getMinSelectionIndex();
				String ixnID = outboundGui.getSelectedInteractionID(selectedRow);                

				sLog.debug("Selection in the interaction table "+ixnID);

				if(ixnID !=null)
					selectedInteraction = (InteractionVoice) interactionIds.get(ixnID);

				updateSelectedReferences();
			}  
		}
	}

	/**
	 * Handles selection in the record table of the outbound chain.
	 *
	 */
	public class ChainedRecordSelectionListener implements ListSelectionListener {
		JTable table;

		// It is necessary to keep the table since it is not possible
		// to determine the table from the event's source
		public ChainedRecordSelectionListener(JTable table) {
			this.table = table;
		}
		public void valueChanged(ListSelectionEvent e) {
			// If cell selection is enabled, both row and column change events are fired

			ListSelectionModel lsm = (ListSelectionModel)e.getSource();
			if (!lsm.isSelectionEmpty()) {
				int selectedRow = lsm.getMinSelectionIndex();
				String recordID = outboundGui.getSelectedChainedRecord(selectedRow);                

				sLog.debug("Selection in the chained record table "+recordID);

				if(!recordID.equals("") && selectedChain!=null)
				{
					try {
						selectedChain.setActiveRecord(recordID);
					} catch (RequestFailedException e1) {
						sLog.error("setActiveRecord "+e1.getLocalizedMessage());
					}
					updateSelectedReferences();
				}
			}  
		}
	}
	
	private OutboundRecord.CallResult getCallResultFromString(String val)
	{
		OutboundRecord.CallResult result =OutboundRecord.CallResult.UNKNOWN;
		for(int i=0; i<OutboundRecord.CallResult.max(); i++)
		{
			if(OutboundRecord.CallResult.getCallResult(i)!= null && OutboundRecord.CallResult.getCallResult(i).toString().compareTo(val)==0)
				return OutboundRecord.CallResult.getCallResult(i);
		}
		return result;
	}

	private OutboundRecord.PhoneType getPhoneTypeFromString(String val)
	{
		OutboundRecord.PhoneType result = OutboundRecord.PhoneType.NO_TYPE;
		for(int i=0; i<OutboundRecord.CallResult.max(); i++)
		{
			if( OutboundRecord.PhoneType.getPhoneType(i)!= null &&  OutboundRecord.PhoneType.getPhoneType(i).toString().compareTo(val)==0)
				return OutboundRecord.PhoneType.getPhoneType(i);
		}
		return result;
	}

	private OutboundChain.TreatmentType getTreatmentTypeFromString(String val)
	{
		OutboundChain.TreatmentType result = OutboundChain.TreatmentType.NONE;
		for(int i=0; i<OutboundChain.TreatmentType.max(); i++)
		{
			if( OutboundChain.TreatmentType.getTreatmentType(i)!= null &&  OutboundChain.TreatmentType.getTreatmentType(i).toString().compareTo(val)==0)
				return OutboundChain.TreatmentType.getTreatmentType(i);
		}
		return result;
	}

	/**
	 * Handles selection in comboboxes.
	 *
	 */
	public class ComboBoxListener implements ActionListener
	{
		public void actionPerformed(ActionEvent e) {

			if(e.getSource() == callResultComboBox && selectedRecord != null)
			{
				String result = (String) callResultComboBox.getSelectedItem();				
				try {
					selectedRecord.setCallResult(getCallResultFromString(result));
					selectedRecord.update();
				} catch (RequestFailedException e1) {
					sLog.error("setCallResult + update failed " +e1.getLocalizedMessage() );
				}
				sLog.debug("ComboBoxListener - Call result is now: "+selectedRecord.getCallResult());

			}else if(e.getSource() == phoneTypeComboBox && selectedRecord != null)
			{
				try {
					selectedRecord.setPhoneType(getPhoneTypeFromString(phoneTypeComboBox.getSelectedItem().toString()));
					selectedRecord.update();
				} catch (RequestFailedException e1) {
					sLog.error("setPhoneType + update failed " +e1.getLocalizedMessage() );
				}
				sLog.debug("ComboBoxListener - Phone type is now: "+selectedRecord.getPhoneType());
			}
			else if(e.getSource() == chainTreatmentComboBox && selectedChain != null)
			{
				selectedChain.setTreatment(getTreatmentTypeFromString(chainTreatmentComboBox.getSelectedItem().toString()));            	
				sLog.debug("ComboBoxListener - chain treatment is now: "+selectedChain.getTreatment().toString());           
			}
		}
	}
}