/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2006 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
/**
Queued Interaction Java API 7.5 Code Examples
*/
package queued.sdk.java.examples;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JList;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;


import com.genesyslab.omsdk.qil.QILFactory;
import com.genesyslab.omsdk.qil.QILQueue;
import com.genesyslab.omsdk.qil.event.QILQueueContentChangedEvent;
import com.genesyslab.omsdk.qil.event.QILQueueEvent;
import com.genesyslab.omsdk.qil.event.QILQueueListener;
import com.genesyslab.omsdk.qil.exception.QILRequestFailedException;
import com.genesyslab.omsdk.qil.exception.QILUninitializedException;

/**
 * Monitors queues and fires alarm when queues reach their interaction threshold.
 * <p>This example uses process bars to display queue activity:<br>
 * activity = (number of interactions in the queue)/(alarm threshold)*100</p>
 * <p>The alarm threshold is the number of interactions that you allow in the queue. 
 * You can set up a default threshold for all queues or specify a particular interaction threashold for each queue.
 * The GUI sends an alarm if the number of interactions
 * in this queue is greater than the threshold.
 * When the queue contains exactly opr more interaction than the threshold, an alarm is fired:
 * <ul><li>A message is displayed in the log console.</li>
 * <li> The total number of fired alarms is increased.</li>
 * <li>The queue name is added to the list of queues for which an alarm was fired, and the details for this queue include alarm time information. </li>
 * </ul>
 * </p>
 * <p>This example uses the following Queued Interaction Java Application Blocks:<ul>
 * <li><code>GetQueue</code> Application Block</li>
 * </ul>
 * </p>
 * @version 7.5.001.01
 */
public class MultipleAlarm implements QILQueueListener, ItemListener {
	/** Connection to the Genesys servers. */
	SimpleConnector connection;
	public HashMap queueInfo;
	AlarmGui gui;
    public JTextField thresholdField;
    public JButton applyThresholdButton;
    public JComboBox queueComboBox; 
    public JButton applyDefaultThresholdButton;
    public JTextField defaultThresholdField;
 
	/**
	 * Constructor. Connects to Configuration Server and Interaction Server.
	 */
	public MultipleAlarm(String windowTitle)
	{
		connection = new SimpleConnector();
		gui = new AlarmGui(windowTitle);
        linkWidgetsToGui();
        gui.mainGuiWindow.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        gui.mainGuiWindow.addWindowListener(connection);
        gui.mainGuiWindow.pack();
        gui.mainGuiWindow.setVisible(true);
        gui.addQueueComboxBoxItemListener(this);
        gui.alarmListener = new AlarmListSelectionListener();
        gui.warningListener = new WarningListSelectionListener();

	}
	/**
	 * Links QIL information to GUI components.
	 *<p> This method uses the <code>GetQueue</code> Application Block to retrieve the list of queues to be monitored.</p> 
	 */
	public void linkWidgetsToGui()
	{
		
		queueInfo= new HashMap();
		/* Source from the GetQueue Application Block, refer to the getAllQueues() method */
		try{
			Collection queues = QILFactory.getQILFactory().getAllQueues();
            queues.iterator();
            for (Iterator iterator = queues.iterator(); iterator.hasNext();) {
                QILQueue queue = (QILQueue) iterator.next();
                queueInfo.put (queue.getID(), new Integer[]{new Integer(0), new Integer(20)} );
                gui.setQueueInfo(queue.getID(), 0,0 );
                queue.addQueueListener(this);
                queue.startMonitoring();
    			
            }
		}		
		catch (QILRequestFailedException __e)
		{
			gui.writeLogMessage("Request failed. Connection to Configuration Server may be lost. "+__e.toString(), AlarmGui.errorStyle);
		}
		catch (QILUninitializedException e) {
            //  Exception thrown if QIL Factory was not initialized
			gui.writeLogMessage("Request failed. QILFactory is not initialized. "+e.toString(), AlarmGui.errorStyle);
			
        } 
		

		thresholdField = gui.thresholdField;
		applyThresholdButton = gui.applyThresholdButton;
		queueComboBox = gui.queueComboBox;
		
		applyThresholdButton.setAction(new AbstractAction("Apply Threshold") {
			public void actionPerformed(ActionEvent actionEvent) {
				String selectedQueueName = (String) queueComboBox.getSelectedItem();
				if(selectedQueueName != null && queueInfo.containsKey(selectedQueueName))
				{
					Integer newThreshold = new Integer(thresholdField.getText());
					Integer[] info = (Integer[]) queueInfo.get(selectedQueueName);
					info[1] = newThreshold;
					queueInfo.put(selectedQueueName,info);           
					
					int activity = getActivity(selectedQueueName);
					gui.setQueueInfo(selectedQueueName, (info[0]).intValue(), activity);
					if(activity >= 100)
						gui.writeLogMessage(" !!! Alarm on "+ selectedQueueName+ ": "+ (info[0]).intValue() +" ixns for a threshold of "+newThreshold+" ixns !!!", AlarmGui.errorStyle);
					
				}		
			}
		});
		
		defaultThresholdField = gui.defaultThresholdField;
		defaultThresholdField.setText("20");

		applyDefaultThresholdButton = gui.applyDefaultThresholdButton;
		applyDefaultThresholdButton.setAction(new AbstractAction("Apply Threshold") {
			public void actionPerformed(ActionEvent actionEvent) {
				{
					Integer defaultThreshold = new Integer(defaultThresholdField.getText());
					setDefaultThreshold(defaultThreshold);
					
				}
			}
			});
		
	}
	
	/**
	 * Sets up a default interaction threshold for all queues.
	 * <p>This method modifies the threshold associated with queues, and updates the queues activity.</p>
	 * @param defaultThreshold
	 */
	public void setDefaultThreshold(Integer defaultThreshold)
	{
		Iterator queueNames = queueInfo.keySet().iterator();
		Iterator queueInfoIxns = queueInfo.values().iterator();
		
		while(queueNames.hasNext())
		{
			String queueName=(String) queueNames.next();
			Integer[] infoIxns = (Integer[]) queueInfoIxns.next();
			
			int nbInteractions = infoIxns[0].intValue() ;
			infoIxns[1] = defaultThreshold;
			queueInfo.put(queueName,infoIxns );
			int activity = getActivity(queueName);
			gui.setQueueInfo(queueName, nbInteractions, activity);
			this.thresholdField.setText(infoIxns[1].toString());
			if(activity >= 100)
				gui.writeLogMessage(" !!! Alarm on "+ queueName+ ": "+ (infoIxns[0]).intValue() +" ixns for a threshold of "+infoIxns[1].toString()+" ixns !!!", AlarmGui.errorStyle);
			
		}
	}
	
	
	public static void main(String[] args) {
		  //Schedule a job for the event-dispatching thread:
      //creating and showing this application's GUI.
      SwingUtilities.invokeLater(new Runnable() {
          public void run() {
          	MultipleAlarm application = new MultipleAlarm(
                      "Queued Interaction Java API: Multiple Alarm");
          }
      });
	}

	/**
	 * Receives <code>QILQueueEvent</code>, which report changes in the monitored queue.
	 * If you implement the <code>QILQueueListener</code> interface, you must implement this method.
	 * <br>The <code>SimpleQueueAlarm</code> example uses this method to manage the queue alarm.
	 *
	 */
	public void handleQueueEvent(QILQueueEvent event) {
		
		QueueEventThread p = new QueueEventThread(event);
		p.start();
	}
	/**
	 * Updates the Service table that lists available services, and to log these ServiceEvents as they occur.
	 */
	class QueueEventThread extends Thread
	{
		QILQueueEvent event;
		
		public QueueEventThread(QILQueueEvent _event)
		{
			event=_event;			
		}
		/////////////// Source from the MyQILQueueListener Application Block /////////////
		public void run()
		{
			
			if(event instanceof QILQueueContentChangedEvent)
			{
				QILQueueContentChangedEvent changeEvent = (QILQueueContentChangedEvent)event;
				String queueName = event.getQueue().getID();
				Integer[] infoIxns = (Integer[]) queueInfo.get(queueName);
				int nbInteractions = infoIxns[0].intValue() + changeEvent.getAddedInteractions().size()
				- changeEvent.getRemovedInteractions().size();
				int threshold = infoIxns[1].intValue();
				infoIxns[0]= new Integer(nbInteractions);
				queueInfo.put(queueName,infoIxns );
				int activity = getActivity(queueName);
				gui.writeLogMessage("   "+ queueName+ " content changed:  "
						+ nbInteractions +" ixn(s) "+ " threshold (" + threshold + ")", AlarmGui.queueEventStyle);
				
				gui.setQueueInfo(queueName, nbInteractions, activity);
				if(activity >= 100)
					gui.writeLogMessage(" !!! Alarm on "+ queueName+ ": "+ nbInteractions +" ixns for a threshold of "+threshold +" ixns !!!", AlarmGui.errorStyle);
				
			}
		}
	}

	/** Implements the <code>ItemListener</code> method.
	 * <p> If the user selects a queue in the combox, this method updates the <code>Queue Details</code> panel.
	 * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
	 */
	public void itemStateChanged(ItemEvent evt) {
		if (evt.getStateChange() == ItemEvent.SELECTED) {
			String queueName = (String) evt.getItem();
			Integer[] infoIxns = (Integer[]) queueInfo.get(queueName);
			int nbInteractions = infoIxns[0].intValue() ;
			gui.setSelectedQueueDetails( queueName, nbInteractions, getActivity(queueName));
			this.thresholdField.setText(infoIxns[1].toString());
			
		}
		
	}
	/**
	 * Returns the queue activity as a percentage.
	 */
	public int getActivity(String queueName)
	{
		Integer[] infoIxns = (Integer[]) queueInfo.get(queueName);
		int nbInteractions = infoIxns[0].intValue() ;
		int threshold = infoIxns[1].intValue();
		int ratio =  (int) ((nbInteractions * 100) / threshold) ; 
		return ratio;
	}
		
	
	/**
	 * Handles selection in the alarm list of the <code>Alarm Details</code> dialog box.
	 */
	class AlarmListSelectionListener implements ListSelectionListener {
		// This method is called each time the user changes the set of selected items
		public void valueChanged(ListSelectionEvent evt) {
			// When the user release the mouse button and completes the selection,
			// getValueIsAdjusting() becomes false
			if (!evt.getValueIsAdjusting()) {
				JList list = (JList)evt.getSource();
				
				// Get all selected items
				Object[] selected = list.getSelectedValues();
				
				
				// Iterate all selected items
				for (int i=0; i<selected.length; i++) {
					String sel = (String) selected[i];
					Integer[] info = (Integer[]) queueInfo.get(sel);
					gui.setDetailsAboutSelectedAlarm(sel, info[0].intValue(), getActivity(sel));
					
				}
				
				
			}
		}
	}
	/**
	 * Handles selection in the queue list of the <code>Warning Details</code> dialog box.
	 **/
	class WarningListSelectionListener implements ListSelectionListener {
		// This method is called each time the user changes the set of selected items
		public void valueChanged(ListSelectionEvent evt) {
			// When the user release the mouse button and completes the selection,
			// getValueIsAdjusting() becomes false
			if (!evt.getValueIsAdjusting()) {
				JList list = (JList)evt.getSource();
				
				// Get all selected items
				Object[] selected = list.getSelectedValues();
				
				
				// Iterate all selected items
				for (int i=0; i<selected.length; i++) {
					String sel = (String) selected[i];
					Integer[] info = (Integer[]) queueInfo.get(sel);
					gui.setDetailsAboutSelectedWarning(sel, info[0].intValue(), getActivity(sel));
					
				}
				
				
			}
		}
	}
}
