/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2006 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
/**
Queued Interaction Java API 7.5 Code Examples
*/
package queued.sdk.java.examples;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.UIManager;
import javax.swing.border.TitledBorder;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.SwingUtilities;

import javax.swing.event.ListSelectionListener;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Vector;
import java.util.Map;
import java.util.Iterator;

import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.text.BadLocationException;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledDocument;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;


/**
 * Graphical user interface for the  
 * Queued Inetraction Java code examples.
 *
 * <p>All of the widgets needed by the examples are created before the GUI is
 * displayed, but this interface enables them only if they are needed by the example that
 * is being run.
 * @version 7.5.001.00
 */
public class OpenMediaSdkGui {

	// Colors
    
    public Color obscureWeakGreen = new Color(0, 51, 0);
    public Color darkFadedGreen = new Color(0, 153, 0);
    public Color obscureDullGreen = new Color(0, 102, 0);
    public Color paleWeakCyan = new Color(204, 255, 255);
    public Color lightWeakGreen = new Color(153, 204, 153);
    public Color paleWeakGreen = new Color(204, 255, 204);
    public Color paleWeakYellow = new Color(255, 255, 204);
    public Color paleWeakBlue = new Color(153,204,255);
    public Color paleWeakRed = new Color(255, 204, 204);
    
    public Color activeBackgroundColor = paleWeakYellow;
    public Color inactiveColor = Color.lightGray;
    
    public Color configurationEventColor = Color.blue;    
    public Color queueEventColor = darkFadedGreen;
    public Color errorColor = Color.red;
    public Color interactionEventColor = new Color(255,102,0);
    public Color serviceEventColor = obscureWeakGreen;
    public Color selectedLineColor = new Color(2,131,212);
    
   
    public Font bold = new Font("", Font.BOLD, 10);
    public Font activeFont = bold;
    

    // Example types
    static String MONITOR_QUEUE = "SimpleMonitorQueue";
    static String SIMPLE_SERVICE = "SimpleService";
    static String MONITOR_IXN = "SimpleMonitorInteraction";
    static String SUPERVISOR = "SimpleSupervisor";

    // Booleans for needed widgets, depending on the example type
    public boolean needQueueWidget = false;
    public boolean needIxnWidget = false;
    public boolean needSupervisorWidget = false;

    //------- Panels and subcomponents -----------
    // GUI: the frame and the buttons
    public JFrame mainGuiWindow;
    Container contentPane;
    GridBagConstraints gridBagConstraints = new GridBagConstraints();

    //--Service Table
    OpenMediaSdkTableModel serviceTableModel;
    JScrollPane serviceTableScrollPane;
    TitledBorder serviceTableTitledBorder = BorderFactory
    .createTitledBorder(" Services ");

    //--Event Configuration panel
    //service event configuration panel
    JPanel serviceEventLogConfigurationPanel;
    JRadioButton noneServiceEventsRadioButton;
    JRadioButton standardServiceEventsRadioButton;
    JRadioButton debugServiceEventsRadioButton;
    TitledBorder serviceEventLogConfigurationPanelTitledBorder = BorderFactory
    .createTitledBorder("Service Event Log Messages");
    public boolean logServiceEvents = true;
    public boolean createDetailedServiceEventMessages = false;

    //--Queue event configuration panel
    JPanel queueEventLogConfigurationPanel;
    JRadioButton noneQueueEventsRadioButton;
    JRadioButton standardQueueEventsRadioButton;
    JRadioButton debugQueueEventsRadioButton;
    TitledBorder queueEventLogConfigurationPanelTitledBorder = BorderFactory
    .createTitledBorder("Queue Event Log Messages");
    public boolean logQueueEvents = true;
    public boolean createDetailedQueueEventMessages = false;
    

    //--Interaction event configuration panel
    JPanel ixnEventLogConfigurationPanel;
    JRadioButton noneIxnEventsRadioButton;
    JRadioButton standardIxnEventsRadioButton;
    JRadioButton debugIxnEventsRadioButton;
    TitledBorder ixnEventLogConfigurationPanelTitledBorder = BorderFactory
    .createTitledBorder("Interaction Event Log Messages");
    public boolean logIxnEvents = true;
    public boolean createDetailedIxnEventMessages = false;


    //--Log area
    JScrollPane logPaneScrollPane;
    JTextPane logPane;
    Calendar calendar = new GregorianCalendar();
    DateFormat timestamp = new SimpleDateFormat("yyyy-dd-MM HH:mm:ss:SSS");
    StyledDocument logDocument;
    String newline = "\n";
    public static String serviceEventStyle = "serviceEvent";
    public static String interactionEventStyle = "interactionEvent";
    public static String queueEventStyle = "queueEvent";
    public static String errorStyle = "error";
    

    //--------- Widgets for managing Queues
    // Queue Table
    OpenMediaSdkTableModel queueTableModel;
    JTable queueTable;
    JScrollPane queueTableScrollPane;
    TitledBorder queueTableTitledBorder = BorderFactory
    .createTitledBorder(" Queues ");

    // Monitor buttons
    JPanel monitorQueuePanel;
    JButton startButton;
    JButton stopButton;
    TitledBorder monitorButtonsTitledBorder = BorderFactory
    .createTitledBorder(" Monitor Buttons ");
    
    // Tree of Annex (key-value pairs)
    JScrollPane	annexTreeScrollPane;
    JTree annexTree;
    TitledBorder annexTreeTitledBorder = BorderFactory
    .createTitledBorder(" Queue Information ");

    //  --------- Widgets for managing Interactions
    // Ixn Table
    OpenMediaSdkTableModel ixnTableModel;
    HashMap ixnModels ;
    JTable ixnTable;
    JScrollPane ixnTableScrollPane;
    TitledBorder ixnTableTitledBorder = BorderFactory
    .createTitledBorder(" Interactions ");
    
    // -------------- Widgets for supervisor
    TitledBorder supervisorTitledBorder = BorderFactory
    .createTitledBorder(" Supervisor ");
    HashMap ixnProperties;
    JPanel supervisorPanel;
    public JButton modeButton ;
    public JLabel modeLabel;
    JTree ixnPropertyTree;
    JScrollPane ixnPropertyScrollPane ;
    TitledBorder selectedIxnTreeTitledBorder = BorderFactory
    .createTitledBorder(" Properties of the selected interaction ");
    JLabel propNameLabel;
    public JTextField propNameTextField;
    JLabel propValueLabel;
    public JTextField propValueTextField;
    
    
    public JButton propButton ;
    public JButton lockButton ;
    public JButton unlockButton ;
    public JButton pullButton ;
    public JButton leaveButton;    
    public JButton stopProcessingButton ;
    public JLabel interactionIDLabel ;

    Dimension buttonPreferredsize = new Dimension(100, 20);
    int spaceUnderTextField = 3;
    int spaceNextToBorder = 1;


    //---------- Constructor and 'create panels' methods.
    /**
     * OpenMediaSdkGui constructor. Creates an <code>OpenMediaSdkGui</code> instance according to the given example type.
     * @param windowTitle
     *            The title of the GUI window for the example that is currently
     *            being run.
     * @param exampleType
     *            The type of code example being executed.
     */
    public OpenMediaSdkGui(
            String windowTitle,
            String exampleType) {
        determineCodeExampleType(exampleType);
        createGui(windowTitle);

    }

    /**
     * Figures out what type of code example is being run.
     *
     * This information is used to determine whether widgets used by that code
     * example will need to be enabled when the GUI is activated.
     *
     * @param exampleType
     *            The type of code example being executed.
     */
    private void determineCodeExampleType(
            String exampleType) {

        if (exampleType.equals(MONITOR_QUEUE)) {
            needQueueWidget = true;
        }
        if (exampleType.equals(MONITOR_IXN)) {
        	needQueueWidget = true;
            needIxnWidget = true;           
        }
        if (exampleType.equals(SUPERVISOR)) {
        	needQueueWidget = true;
            needIxnWidget = true;
            needSupervisorWidget = true;
        }

    }

    /**
     * Sets the Java look and feel to the look and feel of the
     * platform the example is being run on.
     *
     * <br>Then, it creates the main window (JFrame) and content pane being used by
     * the GUI. After that, it calls methods that create the various components
     * of the user interface.
     *
     * @param windowTitle
     *            The title of the GUI window for the example that is currently
     *            being run.
     */
    private void createGui(String windowTitle) {

        try {
            UIManager
                    .setLookAndFeel(UIManager
                            .getSystemLookAndFeelClassName());
        } catch (Exception exception) {
            exception.printStackTrace();
        }

        // Create the main window
        mainGuiWindow = new JFrame(windowTitle);
        mainGuiWindow
                .setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        contentPane = mainGuiWindow
                .getContentPane();
        contentPane
                .setLayout(new GridBagLayout());

        //contentPane.setSize(new Dimension(600,800));
        
        
        ////////////////// CONTENT PANE ///////////////////////
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.BOTH;
        
        c.gridwidth = 1;
        c.gridx = 0;
        c.gridy = 0;
        contentPane.add( createQueueTable(), c);
        
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 0;
        contentPane.add( createAnnexTree(), c);
        
       /* c.weightx = 1;
        c.weighty = 1;*/
        c.gridwidth = 2;
        c.gridx = 0;
        c.gridy = 4;
        contentPane.add( createInteractionTable(), c);
        
        
        JTabbedPane tabbedPane = new JTabbedPane();
        JPanel mainPane = new JPanel(new GridBagLayout());
        
        //////////////// MAIN PANE /////////////
             
        c = new GridBagConstraints();
        c.fill = GridBagConstraints.BOTH;

        // Create and place panels
        c.gridwidth = 2;
        c.gridx = 0;
        c.gridy = 0;
        mainPane.add(createServiceInfoPanel(), c);
        
        c.gridwidth = 1;
        c.gridheight = 1;        
        c.gridx = 1;
        c.gridy = 1;
        mainPane.add(createServiceEventLogConfigurationPanel(),c);

        c.gridx = 2;
        c.gridy = 1;
        mainPane.add(createQueueEventLogConfigurationPanel(),c);
    	
        c.gridx = 3;
        c.gridy = 1;
        mainPane.add(createInteractionEventLogConfigurationPanel(),c);   
        
        tabbedPane.add(mainPane,0);
        tabbedPane.setTitleAt(0, "Main");
        
        //////////////////// Monitoring /////////////////////
        
        JPanel monitoringPane = new JPanel(new GridBagLayout());
        c.fill = GridBagConstraints.BOTH;
        c.gridwidth = 1;
        c.gridheight = 1;
        c.gridx = 0;
        c.gridy = 0;
        monitoringPane.add(createMonitorButtons(),c);
        
        
        c.gridwidth = 2;
        c.gridheight = 2;        
        c.gridx = 1;
        c.gridy = 0;
        monitoringPane.add(createSupervisorPanel(),c);
        
        
        c.gridwidth = 3;
        c.gridheight = 2;
        c.gridx = 3;
        c.gridy = 0;
        monitoringPane.add(createIxnPropertyTree(),c);
        
        
       
    	
        tabbedPane.add(monitoringPane,1);
        tabbedPane.setTitleAt(1, "Monitoring");
    
        c.gridwidth = 2;
        c.gridheight = 1;   
        c.gridx = 0;
        c.gridy = 5;
        contentPane.add(tabbedPane,c);
        
        c.gridwidth = 1;
        c.gridheight = 1; 
        createLogAreaScrollPane();

        //Activate panels depending on the example type
        activateWidgets();

        mainGuiWindow.setResizable(true);


    }
    /**
     * Creates the <code>Service</code> panel displaying the services' status.
     */
    private JScrollPane createServiceInfoPanel()
    {
    	serviceTableModel = new OpenMediaSdkTableModel();
    	serviceTableModel.colName0 = "Service Type";
    	serviceTableModel.colName1 = "Status";
    	serviceTableModel.NUM_COLUMNS = 2;

    	JTable serviceTable = new JTable(serviceTableModel);

    	serviceTable.setSelectionBackground(new Color(2,131,212));
    	serviceTable.setSelectionForeground(new Color(255,255,255));

    	serviceTableScrollPane = new JScrollPane(serviceTable);
    	
    	
    	 
    	serviceTableScrollPane.setBorder(serviceTableTitledBorder);
    	serviceTable.setPreferredScrollableViewportSize(new Dimension(180, 50));
    	serviceTable.setMinimumSize(new Dimension(180, 50));
    	serviceTable.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);


    	return serviceTableScrollPane;
    }

    /**
     * Creates the configuration panel for service events' log messages.
     */
    private JPanel createServiceEventLogConfigurationPanel() {
        serviceEventLogConfigurationPanel = new JPanel();
        serviceEventLogConfigurationPanel.setLayout(new BoxLayout(
                serviceEventLogConfigurationPanel, BoxLayout.X_AXIS));

        serviceEventLogConfigurationPanel.setBorder(BorderFactory
                .createTitledBorder(serviceEventLogConfigurationPanelTitledBorder));

        noneServiceEventsRadioButton = new JRadioButton("None");
        noneServiceEventsRadioButton.setActionCommand("enable");
        noneServiceEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                        logServiceEvents = false;
                        createDetailedServiceEventMessages = false;
                        noneServiceEventsRadioButton.setSelected(true);
                        standardServiceEventsRadioButton.setSelected(false);
                        debugServiceEventsRadioButton.setSelected(false);
                    }
                });
        serviceEventLogConfigurationPanel.add(noneServiceEventsRadioButton);

        standardServiceEventsRadioButton = new JRadioButton("Standard", true);
        standardServiceEventsRadioButton.setActionCommand("enable");
        standardServiceEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                    	logServiceEvents = true;
                        createDetailedServiceEventMessages = false;
                        noneServiceEventsRadioButton.setSelected(false);
                        standardServiceEventsRadioButton.setSelected(true);
                        debugServiceEventsRadioButton.setSelected(false);
                    }
                });
        serviceEventLogConfigurationPanel.add(standardServiceEventsRadioButton);

        debugServiceEventsRadioButton = new JRadioButton("Debug");
        debugServiceEventsRadioButton.setActionCommand("enable");
        debugServiceEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                    	logServiceEvents = true;
                    	createDetailedServiceEventMessages = true;
                    	noneServiceEventsRadioButton.setSelected(false);
                    	standardServiceEventsRadioButton.setSelected(false);
                        debugServiceEventsRadioButton.setSelected(true);
                    }
                });
        serviceEventLogConfigurationPanel.add(debugServiceEventsRadioButton);

        return serviceEventLogConfigurationPanel;
    }


    /**
     * Creates the configuration panel for queue events' log messages.
     */
    private JPanel createQueueEventLogConfigurationPanel() {
        queueEventLogConfigurationPanel = new JPanel();
        queueEventLogConfigurationPanel.setLayout(new BoxLayout(
        		queueEventLogConfigurationPanel, BoxLayout.X_AXIS));
       
        queueEventLogConfigurationPanel.setBorder(BorderFactory
                .createTitledBorder(queueEventLogConfigurationPanelTitledBorder));

        noneQueueEventsRadioButton = new JRadioButton("None");
        noneQueueEventsRadioButton.setActionCommand("enable");
        noneQueueEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                    	logQueueEvents = false;
                        createDetailedQueueEventMessages = false;
                        noneQueueEventsRadioButton.setSelected(true);
                        standardQueueEventsRadioButton.setSelected(false);
                        debugQueueEventsRadioButton.setSelected(false);
                    }
                });
        queueEventLogConfigurationPanel.add(noneQueueEventsRadioButton);

        standardQueueEventsRadioButton = new JRadioButton("Standard", true);
        standardQueueEventsRadioButton.setActionCommand("enable");
        standardQueueEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                    	logQueueEvents = true;
                    	createDetailedQueueEventMessages = false;
                    	noneQueueEventsRadioButton.setSelected(false);
                        standardQueueEventsRadioButton.setSelected(true);
                        debugQueueEventsRadioButton.setSelected(false);
                    }
                });
        queueEventLogConfigurationPanel.add(standardQueueEventsRadioButton);

        debugQueueEventsRadioButton = new JRadioButton("Debug");
        debugQueueEventsRadioButton.setActionCommand("enable");
        debugQueueEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                    	logQueueEvents = false;
                    	createDetailedQueueEventMessages = true;
                    	noneQueueEventsRadioButton.setSelected(false);
                    	standardQueueEventsRadioButton.setSelected(false);
                    	debugQueueEventsRadioButton.setSelected(true);
                    }
                });
        queueEventLogConfigurationPanel.add(debugQueueEventsRadioButton);

        return queueEventLogConfigurationPanel;
    }
    
    /**
     * Creates the configuration panel for Interaction events' log messages.
     */
    private JPanel createInteractionEventLogConfigurationPanel() {
        ixnEventLogConfigurationPanel = new JPanel();
        ixnEventLogConfigurationPanel.setLayout(new BoxLayout(
        		ixnEventLogConfigurationPanel, BoxLayout.X_AXIS));
        
        ixnEventLogConfigurationPanel.setBorder(ixnEventLogConfigurationPanelTitledBorder);

        noneIxnEventsRadioButton = new JRadioButton("None");
        noneIxnEventsRadioButton.setActionCommand("enable");
        noneIxnEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                    	logIxnEvents = false;
                        createDetailedIxnEventMessages = false;
                        noneIxnEventsRadioButton.setSelected(true);
                        standardIxnEventsRadioButton.setSelected(false);
                        debugIxnEventsRadioButton.setSelected(false);
                    }
                });
        ixnEventLogConfigurationPanel.add(noneIxnEventsRadioButton);

        standardIxnEventsRadioButton = new JRadioButton("Standard", true);
        standardIxnEventsRadioButton.setActionCommand("enable");
        standardIxnEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                    	logIxnEvents = true;
                    	createDetailedIxnEventMessages = false;
                    	noneIxnEventsRadioButton.setSelected(false);
                        standardIxnEventsRadioButton.setSelected(true);
                        debugIxnEventsRadioButton.setSelected(false);
                    }
                });
        ixnEventLogConfigurationPanel.add(standardIxnEventsRadioButton);

        debugIxnEventsRadioButton = new JRadioButton("Debug");
        debugIxnEventsRadioButton.setActionCommand("enable");
        debugIxnEventsRadioButton
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent event) {
                    	logIxnEvents = false;
                    	createDetailedIxnEventMessages = true;
                    	noneIxnEventsRadioButton.setSelected(false);
                    	standardIxnEventsRadioButton.setSelected(false);
                    	debugIxnEventsRadioButton.setSelected(true);
                    }
                });
        ixnEventLogConfigurationPanel.add(debugIxnEventsRadioButton);

        return ixnEventLogConfigurationPanel;
    }

    /**
     * Adds the panel for events' level of detail.
     */
    private JPanel createButtonsPanel()
    {
    	JPanel  wholePanel = new JPanel();
    	wholePanel.setLayout(new BoxLayout(	wholePanel, BoxLayout.X_AXIS));
    	
    	wholePanel.add(createMonitorButtons());
    	
    	serviceEventLogConfigurationPanel = createServiceEventLogConfigurationPanel();    
    	wholePanel.add(serviceEventLogConfigurationPanel);

    	queueEventLogConfigurationPanel = createQueueEventLogConfigurationPanel();
    	wholePanel.add(queueEventLogConfigurationPanel);
    	
    	ixnEventLogConfigurationPanel = createInteractionEventLogConfigurationPanel();
    	wholePanel.add(ixnEventLogConfigurationPanel);
    	return wholePanel;
    }
    /**
     * Activates and colors panels, depending on boolean for needed widgets.
     */
    private void activateWidgets()
    {
    	//Service widgets
    	serviceTableScrollPane.setBackground(activeBackgroundColor);
    	serviceEventLogConfigurationPanel.setBackground(activeBackgroundColor);
    	noneServiceEventsRadioButton.setBackground(activeBackgroundColor);
	    standardServiceEventsRadioButton.setBackground(activeBackgroundColor);
	    debugServiceEventsRadioButton.setBackground(activeBackgroundColor);

    	serviceEventLogConfigurationPanelTitledBorder.setTitleFont(activeFont);
    	serviceTableTitledBorder.setTitleFont(activeFont);
    	serviceEventLogConfigurationPanel.setFont(activeFont);
    	noneServiceEventsRadioButton.setFont(activeFont);
	    standardServiceEventsRadioButton.setFont(activeFont);
	    debugServiceEventsRadioButton.setFont(activeFont);

    	//Queue widgets
	    if(needQueueWidget)
	    {
	    	//Queue event log config
	    	queueEventLogConfigurationPanelTitledBorder.setTitleFont(activeFont);
	    	
	    	queueEventLogConfigurationPanel.setBackground(activeBackgroundColor);
	    	noneQueueEventsRadioButton.setBackground(activeBackgroundColor);
	    	standardQueueEventsRadioButton.setBackground(activeBackgroundColor);
	    	debugQueueEventsRadioButton.setBackground(activeBackgroundColor);
	    	
	    	noneQueueEventsRadioButton.setFont(activeFont);
	    	standardQueueEventsRadioButton.setFont(activeFont);
	    	debugQueueEventsRadioButton.setFont(activeFont);

	    	//Queue Table
	    	queueTableScrollPane.setBackground(activeBackgroundColor);	  
	    	queueTableTitledBorder.setTitleFont(activeFont);
	    	queueTable.setRowSelectionAllowed(true);

	    	//Queue monitor buttons
	    	monitorQueuePanel.setBackground(activeBackgroundColor);
	    	monitorButtonsTitledBorder.setTitleFont(activeFont);
	    	startButton.setFont(activeFont);
	    	stopButton.setFont(activeFont);

	    	
	    	//Queue annex
	    	annexTreeScrollPane.setBackground(activeBackgroundColor);
	    	annexTreeTitledBorder.setTitleFont(activeFont);

	    } else
	    {
	    	queueEventLogConfigurationPanel.setForeground(inactiveColor);
	    	
	    	noneQueueEventsRadioButton.setForeground(inactiveColor);
	    	standardQueueEventsRadioButton.setForeground(inactiveColor);
	    	debugQueueEventsRadioButton.setForeground(inactiveColor);
	    	queueEventLogConfigurationPanelTitledBorder.setTitleColor(inactiveColor);
	    	
	    	
	    	queueTableScrollPane.setForeground(inactiveColor);
	    	queueTable.setForeground(inactiveColor);
	    	 	
	    	monitorQueuePanel.setForeground(inactiveColor);
	    	monitorButtonsTitledBorder.setTitleColor(inactiveColor);
	    	
	    	annexTreeScrollPane.setForeground(inactiveColor);
	    	annexTreeTitledBorder.setTitleColor(inactiveColor);
	    	
	    	queueTable.setForeground(inactiveColor);
	    	queueTableTitledBorder.setTitleColor(inactiveColor);
	    	
	    	queueTable.setRowSelectionAllowed(false);
	   		queueTable.setToolTipText("Launch a \'Simple Monitor Queue\' application to enable selection in this table.");

	    	
	    }
	    if(needIxnWidget)
	    {
	    	ixnEventLogConfigurationPanel.setBackground(activeBackgroundColor);
	    	ixnEventLogConfigurationPanelTitledBorder.setTitleFont(activeFont);
	    	noneIxnEventsRadioButton.setBackground(activeBackgroundColor);
	    	noneIxnEventsRadioButton.setFont(activeFont);
	    	
	    	standardIxnEventsRadioButton.setBackground(activeBackgroundColor);
	    	standardIxnEventsRadioButton.setFont(activeFont);
	    	
	    	debugIxnEventsRadioButton.setBackground(activeBackgroundColor);
	    	debugIxnEventsRadioButton.setFont(activeFont);

	    	//Ixn Table
	    	ixnTableScrollPane.setBackground(activeBackgroundColor);
	    	ixnTableTitledBorder.setTitleFont(activeFont);
	    	
	    } else {
	    	ixnEventLogConfigurationPanelTitledBorder.setTitleColor(inactiveColor);
	    	noneIxnEventsRadioButton.setForeground(inactiveColor);
	    	standardIxnEventsRadioButton.setForeground(inactiveColor);
	    	debugIxnEventsRadioButton.setForeground(inactiveColor);
	    	
	    	ixnTableTitledBorder.setTitleColor(inactiveColor);
	    	
	    	ixnTable.setToolTipText("Launch a \'Simple Monitor Interaction\' application to activate this table.");

	    }
	    if(needSupervisorWidget)
	    {
	    	 supervisorTitledBorder.setTitleFont(activeFont);
	    	 ixnPropertyScrollPane.setBackground(activeBackgroundColor);
	    	 supervisorPanel.setBackground(activeBackgroundColor);
			 modeButton.setEnabled(true) ;
			 modeButton.setFont(activeFont);
			 modeLabel.setFont(activeFont);
			 interactionIDLabel.setFont(activeFont);
			 selectedIxnTreeTitledBorder.setTitleFont(activeFont);
			 propNameLabel.setFont(activeFont);
			 propNameTextField.setEditable(true);
			 propValueLabel.setFont(activeFont);
			 propValueTextField.setEditable(true);
			 propButton.setEnabled(false) ;
			 propButton.setFont(activeFont);
			 lockButton.setEnabled(false);
			 lockButton.setFont(activeFont);
			 unlockButton.setEnabled(false);
			 unlockButton.setFont(activeFont);
			 pullButton.setEnabled(false);
			 pullButton.setFont(activeFont);
			 leaveButton.setEnabled(false);
			 leaveButton.setFont(activeFont);
			 stopProcessingButton.setEnabled(false); 
			 stopProcessingButton.setFont(activeFont);
	    }else{
	    	 supervisorTitledBorder.setTitleColor(inactiveColor);
			 modeButton.setEnabled(false) ;
			 interactionIDLabel.setForeground(inactiveColor);
			 modeLabel.setForeground(inactiveColor);
			 selectedIxnTreeTitledBorder.setTitleColor(inactiveColor);
			 propNameLabel.setForeground(inactiveColor);
			 propNameTextField.setEditable(false);
			 propValueLabel.setForeground(inactiveColor);
			 propValueTextField.setEditable(false);
			 propButton.setEnabled(false) ;
			 lockButton.setEnabled(false);
			 unlockButton.setEnabled(false);			 
			 pullButton.setEnabled(false);
			 leaveButton.setEnabled(false);
			 stopProcessingButton.setEnabled(false); 
	    }
	   
    }
    /**
     * Creates the table displaying the available queues.
     * <br>Includes a renderer class to highlight 
     */
    private JScrollPane createQueueTable()
    {
    	queueTableModel = new OpenMediaSdkTableModel();
    	queueTableModel.NUM_COLUMNS = 3;
    	queueTableModel.colName0 = "Queue Name";
    	queueTableModel.colName1 = "Monitored";
    	queueTableModel.colName2 = "Interactions";

        queueTable = new JTable()
		{
        	public Component prepareRenderer(TableCellRenderer renderer,
        			int rowIndex, int vColIndex) {
        		Component c = super.prepareRenderer(renderer, rowIndex, vColIndex);
        		OpenMediaSdkTableModel model = (OpenMediaSdkTableModel) this.getModel();
        		
        		if(this.getSelectedRow()!= rowIndex && ((String) model.getValueAt(rowIndex,1)).equals(new String("True")))
        		{
        			c.setBackground(activeBackgroundColor);
        			c.setForeground(Color.BLACK);
        		} else if( this.getSelectedRow() == rowIndex){
        			c.setBackground(selectedLineColor);
        			c.setForeground(Color.WHITE);
        		}
        		else
        		{
        				//     If not shaded, match the table's background
        				c.setBackground(getBackground());
        				c.setForeground(Color.BLACK);
        		}
        		
        		return c;
        	}
		};
        queueTable.setModel(queueTableModel);
    	queueTable.setSelectionBackground(new Color(2,131,212));
    	queueTable.setSelectionForeground(new Color(255,255,255));
        queueTable.setPreferredScrollableViewportSize(new Dimension(500, 400));
//      Set the first visible column to 100 pixels wide
        int vColIndex = 0;
        TableColumn col = queueTable.getColumnModel().getColumn(vColIndex);
        int width = 200;
        col.setPreferredWidth(width);

        queueTable.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);

        queueTableScrollPane = new JScrollPane(queueTable);      
        queueTableScrollPane.setBorder(queueTableTitledBorder);
        queueTableScrollPane.setMinimumSize(new Dimension(350, 200));

        return queueTableScrollPane;

    }
    /**
     * Creates the table displaying a list of interactions.
     */
    private JScrollPane createInteractionTable()
    {
    	ixnTableModel = new OpenMediaSdkTableModel();
    	ixnTableModel.NUM_COLUMNS = 6;
    	ixnTableModel.colName0 = "ID";
    	ixnTableModel.colName1 = "Media";
    	ixnTableModel.colName2 = "Type";
    	ixnTableModel.colName3 = "Subtype";
    	ixnTableModel.colName4 = "Status";
    	ixnTableModel.colName5 = "Subject";

    	ixnModels = new HashMap();
    	
        ixnTable = new JTable(ixnTableModel);
        ixnTable.setSelectionBackground(new Color(2,131,212));
        ixnTable.setSelectionForeground(new Color(255,255,255));

//      Set the first visible column to 200 pixels wide
        
        TableColumn col = ixnTable.getColumnModel().getColumn(5);
        col.setPreferredWidth(200);
        col = ixnTable.getColumnModel().getColumn(0);
        col.setPreferredWidth(150);
                
        ixnTableScrollPane = new JScrollPane(ixnTable);
        ixnTableScrollPane.setBorder(ixnTableTitledBorder);
        ixnTableScrollPane.setMinimumSize(new Dimension(300, 200));
        ixnTableScrollPane.setPreferredSize(new Dimension(300, 200));

        return ixnTableScrollPane;

    }
    
    /**
     * Creates a tree displaying the annex data of a queue.
     */
    private JScrollPane createAnnexTree(){

        annexTree = new JTree(new Vector());
        annexTreeScrollPane = new JScrollPane(annexTree);
        annexTreeScrollPane.setPreferredSize(new Dimension(200, 150));
        annexTreeScrollPane.setMinimumSize(new Dimension(200, 150));
        annexTreeScrollPane.setMaximumSize(new Dimension(200, 150));
        annexTreeScrollPane.setBorder(annexTreeTitledBorder);

        return annexTreeScrollPane;
    }
    
	/**
     * Creates buttons for monitoring a queue.
     */
    private JPanel createMonitorButtons(){

        monitorQueuePanel = new JPanel();
    	monitorQueuePanel.setLayout(new BoxLayout(
    			monitorQueuePanel, BoxLayout.X_AXIS));
    	monitorQueuePanel.setPreferredSize(new Dimension(130, 60));
    	monitorQueuePanel.setMinimumSize(new Dimension(130, 60));
    	monitorQueuePanel.setBorder(monitorButtonsTitledBorder);

        // Add a Start button
     	startButton = new JButton("Start");
     	startButton.setEnabled(false);
     	monitorQueuePanel.add(startButton);

     	  // Add a stop Monitor Queue button
     	stopButton = new JButton("Stop");
     	stopButton.setEnabled(false);
     	monitorQueuePanel.add(stopButton);

     	return monitorQueuePanel;


    }

    /**
     * Sets a selection listener for the <code>Queues</code> table.
     */
    public void activateQueueRowSelection(ListSelectionListener rowListener)
    {
    	ListSelectionModel rowSM = queueTable.getSelectionModel();
        rowSM.addListSelectionListener(rowListener);

    }
    /**
     * Creates the supervisor panel and buttons.
     * @return the created supervisor panel.
     */
    private JPanel createSupervisorPanel()
    {
    	supervisorPanel = new JPanel();
    	ixnProperties = new HashMap();
    	supervisorPanel.setBorder(supervisorTitledBorder);
    	supervisorPanel.setLayout(new GridBagLayout());
    	supervisorPanel.setMinimumSize(new Dimension(250, 200));
        
        
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.BOTH;

        /// About Mode
        modeLabel = new JLabel("    Current Mode is: undefined      ");
        c.gridx = 0;
        c.gridy = 0;
        c.gridwidth = 2;
        supervisorPanel.add(modeLabel, c);
        
        modeButton = new JButton("Change Mode");
        c.gridx = 0;
        c.gridy = 1;
        c.gridwidth = 2;
        supervisorPanel.add(modeButton, c);
        
        JSeparator separator1 = new JSeparator();
        c.gridx = 0;
        c.gridy = 2;
        c.gridwidth = 2;
        supervisorPanel.add(separator1, c);
        
        interactionIDLabel = new JLabel("Interaction ID: ");
        c.gridx = 0;
        c.gridy = 3;
        c.gridwidth = 2;
        supervisorPanel.add(interactionIDLabel, c);
        
        
        lockButton = new JButton("Lock");
        c.gridx = 0;
        c.gridy = 4;
        c.gridwidth = 1;
        supervisorPanel.add(lockButton, c);
        
        unlockButton = new JButton("Unlock");
        c.gridx = 1;
        c.gridy = 4;
        c.gridwidth = 1;        
        supervisorPanel.add(unlockButton, c);
                
        pullButton = new JButton("Pull");
        c.gridx = 0;
        c.gridy = 5;
        c.gridwidth = 1;        
        supervisorPanel.add(pullButton, c);
        
        leaveButton = new JButton("Leave");
        c.gridx = 1;
        c.gridy = 5;
        c.gridwidth = 1;        
        supervisorPanel.add(leaveButton, c);
        
        stopProcessingButton = new JButton("Stop Processing"); 
        stopProcessingButton.setPreferredSize(buttonPreferredsize);
        c.gridx = 2;
        c.gridy = 5;
        c.gridwidth = 1;
        supervisorPanel.add(stopProcessingButton, c);
              
        
        JSeparator separator = new JSeparator();
        c.gridx = 0;
        c.gridy = 7;
        c.gridwidth = 2;
        supervisorPanel.add(separator, c);
        
        propNameLabel = new JLabel("Property Name: ");
        c.gridx = 0;
        c.gridy = 7;
        c.gridwidth = 1;
        supervisorPanel.add(propNameLabel, c);
        
        propNameTextField = new JTextField("");
        c.gridx = 1;
        c.gridy = 7;
        c.gridwidth = 2;
        supervisorPanel.add(propNameTextField, c);
        
        propValueLabel = new JLabel("Property Value: ");
        c.gridx = 0;
        c.gridy = 8;
        c.gridwidth = 1;
        supervisorPanel.add(propValueLabel, c);
        
        propValueTextField = new JTextField(" ");
        c.gridx = 1;
        c.gridy = 8;
        c.gridwidth = 2;
        supervisorPanel.add(propValueTextField, c);
        
        propButton = new JButton("Change Properties");
        c.gridx = 1;
        c.gridy = 9;
        c.gridwidth = 2;
        supervisorPanel.add(propButton,c);
        
        return supervisorPanel;
    }
    
    /**
     * Creates the property tree for the interaction that is selected in the interaction table.
     * @return the created scrollpane.
     */
    private JScrollPane createIxnPropertyTree()
    {
    	ixnPropertyTree = new JTree(new Vector());
    	ixnPropertyScrollPane = new JScrollPane(ixnPropertyTree);
    	ixnPropertyScrollPane.setPreferredSize(new Dimension(250, 150));
    	ixnPropertyScrollPane.setMinimumSize(new Dimension(200, 150));
    	ixnPropertyScrollPane.setBorder(selectedIxnTreeTitledBorder);

        return ixnPropertyScrollPane;
    }
    
    /**
     * Sets a selection listener for the Interactions table.
     */
    public void activateIxnRowSelection(ListSelectionListener rowListener)
    {
    	ListSelectionModel rowSM = ixnTable.getSelectionModel();
        rowSM.addListSelectionListener(rowListener);

    }

    /**
     * Creates a panel for displaying log messages.
     */
    private void createLogAreaScrollPane() {
        logPane = new JTextPane();
        logDocument = logPane.getStyledDocument();
        Style defaultStyle = StyleContext.getDefaultStyleContext().getStyle(
                StyleContext.DEFAULT_STYLE);

        Style regular = logDocument.addStyle("regular", defaultStyle);
        StyleConstants.setFontFamily(defaultStyle, "SansSerif");

        Style style = logDocument.addStyle(serviceEventStyle, regular);
        StyleConstants.ColorConstants.setForeground(style, serviceEventColor);

        style = logDocument.addStyle(interactionEventStyle, regular);
        StyleConstants.ColorConstants.setForeground(style,
                interactionEventColor);

        style = logDocument.addStyle(errorStyle, regular);
        StyleConstants.ColorConstants.setForeground(style,
                errorColor);

        style = logDocument.addStyle(queueEventStyle, regular);
        StyleConstants.ColorConstants.setForeground(style,
        		queueEventColor);

        logPaneScrollPane = new JScrollPane(logPane,
                JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        logPaneScrollPane.setPreferredSize(new Dimension(400, 155));
        logPaneScrollPane.setMinimumSize(new Dimension(10, 10));
        logPane.setEditable(false);
        gridBagConstraints.gridwidth = GridBagConstraints.REMAINDER;
        gridBagConstraints.gridheight = GridBagConstraints.REMAINDER;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.fill = GridBagConstraints.BOTH;
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = -1;
        contentPane.add(logPaneScrollPane, gridBagConstraints);
    }

	/**
	 * Creates a tree node corresponding to a map of data.
	 */
	private void setMapCategory(DefaultMutableTreeNode category, Map pairs )
	{
		if(pairs != null)
		{
			Iterator itKeys = pairs.keySet().iterator();
			Iterator itValues = pairs.values().iterator();
			
			while(itKeys.hasNext())
			{
				String key = itKeys.next().toString();
				Object val = itValues.next();
				if(val ==null || val instanceof String)
				{
					String text = key + " = " + val;
					DefaultMutableTreeNode leaf = new DefaultMutableTreeNode(text);
					category.add(leaf);
					}
				else if (val !=null && val instanceof Map)
				{
					DefaultMutableTreeNode node = new DefaultMutableTreeNode(key);
					setMapCategory(node, (Map) val);
					category.add(node);
				}					
			}
	    }
	}

	/**
	 * Adds or updates a line of the Services table.
	 */
	public void setService(String type, String status){
		// The type is the ID of the service.
		// Duplicated to be displayed.
		serviceTableModel.update(new String[] {type,type, status}) ;
	}

	/**
	 * Adds or updates a line of the Queues table.
	 */
	public void setQueue(String name, boolean monitored, String number){
		// The queue name is ID of the queue.
		// (Duplicated to be displayed.)
		 queueTableModel.update(new String[]{name, name, getString(monitored), number}) ;
		
	}

	/**
	 * Gets the name of a queue in the Queues table.
	 */
	public String getQueueNameAt(int row){
		return ((String) queueTableModel.getIDAt(row));
	}
    /**
     * Removes a queue from the Queues table.
     */
    public void removeQueue(String name){
    	queueTableModel.removeID(name) ;
    }

    /**
	 * Updates the Annex tree (Queue information) with new queue data to display.
	 */
	public void setAnnex(String queueName, Map data){

		ixnTableTitledBorder.setTitle("Interactions in queue \'"+queueName+"\'");
		this.ixnTableScrollPane.repaint();
		DefaultMutableTreeNode top =
	        new DefaultMutableTreeNode("Categories");


	    Iterator itCategory = data.keySet().iterator();
	    Iterator itMapCategories = data.values().iterator();

		while(itCategory.hasNext())
		{
			String categoryName = (String) itCategory.next();
			DefaultMutableTreeNode categoryNode = new DefaultMutableTreeNode(categoryName);
			Map categoryPairs = (Map) itMapCategories.next();
			setMapCategory(categoryNode, categoryPairs);
		    top.add(categoryNode);
		}

	    annexTree.setModel(new DefaultTreeModel(top));
	    TreeNode root = (TreeNode)annexTree.getModel().getRoot();
	    
	    // Traverse tree from root
	    expandAll(annexTree, new TreePath(root), true);
	}

	private void expandAll(JTree tree, TreePath parent, boolean expand) {
        // Traverse children
        TreeNode node = (TreeNode)parent.getLastPathComponent();
        if (node.getChildCount() >= 0) {
            for (Enumeration e=node.children(); e.hasMoreElements(); ) {
                TreeNode n = (TreeNode)e.nextElement();
                TreePath path = parent.pathByAddingChild(n);
                expandAll(tree, path, expand);
            }
        }
    
        // Expansion or collapse must be done bottom-up
        if (expand) {
            tree.expandPath(parent);
        } else {
            tree.collapsePath(parent);
        }
    }

    /**
     * Gets a string for a boolean.
     */
    public String getString(boolean toGet){
    	if(toGet)
    		return "True";
    	else
    		return "False";
    }
    
    /**
	 * Gets the ID of an interaction in the Interactions table.
	 */
	public String getInteractionNameAt(String queueName, int row){
		OpenMediaSdkTableModel myModel = (OpenMediaSdkTableModel) ixnModels.get(queueName);
		return ((String) myModel.getIDAt(row));
	}
	
    /**
     * Removes an interaction from the Interactions table.
     */
    public void removeIxn(String queueName, String id){
    	if(ixnModels.containsKey(queueName))
		{
			OpenMediaSdkTableModel myModel = (OpenMediaSdkTableModel) ixnModels.get(queueName);
			myModel.removeID(id) ;
		}
    }

    
	/**
	 * Adds or updates a line of an Interaction table.
	 */
	public void setIxn(String queueName, String ID, 
			String media, String type, String subtype, String status, Map properties)
	{
		if(ixnModels.containsKey(queueName))
		{
			OpenMediaSdkTableModel myModel = (OpenMediaSdkTableModel) ixnModels.get(queueName);
			String subject = "none";
			if( properties.containsKey("Subject"))
			{
				subject = (String) properties.get("Subject");				
			}
					
			myModel.update(new String[]{ID, ID, media, type, subtype, status, subject}) ;
			ixnModels.put(queueName, myModel);
		}
		
		if(this.ixnProperties.containsKey(queueName))
		{
			HashMap ixnMap = (HashMap) ixnProperties.get(queueName);
			ixnMap.put(ID, properties);
		}else
		{
			HashMap newIxnMap = new HashMap();
			newIxnMap.put(ID, properties);
			ixnProperties.put(queueName, newIxnMap);
		}
	}


	/**
	 * Returns interaction properties.
	 */
	public Map getIxnProperties(String queueName, String ID){
		if(this.ixnProperties.containsKey(queueName))
		{
			HashMap ixnMap = (HashMap) ixnProperties.get(queueName);
			return (Map) ixnMap.get(ID);
		}
		return null;
		
	}    
	
	/**
	 * Removes interaction properties.
	 */
	public void removeIxnProperties(String queueName, String ID){
		if(this.ixnProperties.containsKey(queueName))
		{
			HashMap ixnMap = (HashMap) ixnProperties.get(queueName);
			ixnMap.remove(ID);
		}
		
	}
	
	/**
	 * Returns interaction properties.
	 */
	public void clearIxnProperties(){
		DefaultMutableTreeNode top =
	        new DefaultMutableTreeNode("Properties");
		this.ixnPropertyTree.setModel(new DefaultTreeModel(top));
		
	}
	 /**
	 * Updates the Properties tree with interaction data to display.
	 */
	public void switchPropertyTree(String queueName, String ID){		
		
		Map data = getIxnProperties(queueName, ID);
		DefaultMutableTreeNode top =
	        new DefaultMutableTreeNode("Properties");

		if(data!=null)
		{

	    Iterator itCategory = data.keySet().iterator();
	    Iterator itMapCategories = data.values().iterator();

		while(itCategory.hasNext())
		{
			String categoryName = (String) itCategory.next();
			DefaultMutableTreeNode categoryNode = new DefaultMutableTreeNode(categoryName);
			Object nodeValue =  itMapCategories.next();
			try
			{
				Map categoryPairs = null;
				if(nodeValue instanceof String)
				{
					categoryNode = new DefaultMutableTreeNode(categoryName+" = "+ nodeValue.toString());
				}else if(nodeValue instanceof Integer)
				{
					categoryNode = new DefaultMutableTreeNode(categoryName+" = "+ ((Integer) nodeValue).toString());
					
				}else if(nodeValue instanceof ArrayList )
				{
					categoryNode = new DefaultMutableTreeNode(categoryName+" = "+ ((ArrayList) nodeValue).toString());
					
				}
				else
				{
					categoryPairs = (Map) nodeValue ;
				}					
				setMapCategory(categoryNode, categoryPairs);
			}catch(Exception __e)
			{
				categoryNode = new DefaultMutableTreeNode(categoryName+" = " +nodeValue.toString());
				
			}
		    top.add(categoryNode);
		}

		this.ixnPropertyTree.setModel(new DefaultTreeModel(top));
	    TreeNode root = (TreeNode)annexTree.getModel().getRoot();
	    
	    // Traverse tree from root
	    expandAll(ixnPropertyTree, new TreePath(root), true);
		}
	}

	/**
	 * Displays the interaction table associated with a queue.
	 * @param queueName the name of the queue
	 */
	public void switchIxnTable(String queueName)
	{
		if(ixnModels.containsKey(queueName))
		{
			ixnTable.setModel((OpenMediaSdkTableModel) ixnModels.get(queueName));
			TableColumn col = ixnTable.getColumnModel().getColumn(5);
	        col.setPreferredWidth(200);
	        col = ixnTable.getColumnModel().getColumn(0);
	        col.setPreferredWidth(150);
		}
		else{
			OpenMediaSdkTableModel myModel = new OpenMediaSdkTableModel();
			myModel.NUM_COLUMNS = 6;
			myModel.colName0 = "ID";
			myModel.colName1 = "Media";
			myModel.colName2 = "Type";
			myModel.colName3 = "Subtype";
			myModel.colName4 = "Status";
			myModel.colName5 = "Subject";
	        
	    	ixnModels.put(queueName,myModel);
	    	ixnTable.setModel(myModel);
	    	
	    	TableColumn col = ixnTable.getColumnModel().getColumn(5);
	        col.setPreferredWidth(200);
	        col = ixnTable.getColumnModel().getColumn(0);
	        col.setPreferredWidth(150);
		}
		
	}
	/**
	 * Clears the interaction table associated with a queue.
	 * @param queueName the name of the queue.
	 */
	public void clearInteractions(String queueName)
	{
		if(ixnModels.containsKey(queueName))
		{
			OpenMediaSdkTableModel myModel = (OpenMediaSdkTableModel) ixnModels.get(queueName);
			myModel.clear();
		}
	}


	/**
	 * Inserts the log message with the accurate style in the log panel.
	 */
	public void writeLogMessage(String eventString, String eventStyle) {
	    try {
	        logDocument.insertString(logDocument.getLength(),
	        		createTimeStamp()+" "+ eventString+newline,
	                logDocument.getStyle(eventStyle));
	        setCaretPosition();
	    } catch (BadLocationException badLocationException) {
	        System.err.println("Couldn't insert text into text pane.");
	    }
	}

	/**
	 * Sets the log pane position at the end of the document.
	 */
	private void setCaretPosition() {
	    SwingUtilities.invokeLater(new Runnable() {
	        public void run() {
	            logPane.setCaretPosition(logDocument.getLength());
	        }
	    });
	}

	/**
	 * Creates the time stamp added to each log message.
	 */
	private String createTimeStamp() {
	    return timestamp.format(new GregorianCalendar().getTime());
	}

	

}