using System;
using System.Collections;
using System.ComponentModel;
using com.genesyslab.ail;
using com.genesyslab.ail.ws.system;
using com.genesyslab.ail.ws._event;
using com.genesyslab.ail.ws.agent;
using com.genesyslab.ail.ws.place;
using com.genesyslab.ail.ws.interaction;
using com.genesyslab.ail.ws.interaction.voice;

namespace AgentExample 
{
	/// <summary>
	/// This class includes the factory and the services used by the application.
	/// It provides a method to connect using the factory. v1.4-01/03/05
	/// </summary>
	public class Connection
	{
		public com.genesyslab.ail.ServiceFactory mServiceFactory = null;
		public com.genesyslab.ail.ws._event.IEventService mEventService = null;
		public com.genesyslab.ail.ws.agent.IAgentService mAgentService = null;
		public com.genesyslab.ail.ws.place.IPlaceService mPlaceService = null;

		/// <summary>
		/// Connects and creates the services.
		/// </summary>
		public Connection()
		{
			try
			{
				mServiceFactory = ServiceFactory.createServiceFactory(null, null,null);
				mEventService = mServiceFactory.createService(typeof(IEventService), null) as IEventService;
				mAgentService = mServiceFactory.createService(typeof(IAgentService), null) as IAgentService;
				mPlaceService = mServiceFactory.createService(typeof(IPlaceService), null) as IPlaceService;
				
			}
			catch(Exception ex)
			{
				mEventService = null;
				mServiceFactory = null;
				throw(ex);
			}
		}

		/// <summary>
		/// Gets the value of a key in a key-value pair array.
		/// </summary>
		/// <param name="keyValue">Array of key-value pairs.</param>
		/// <param name="key">Key to search in the array.</param>
		/// <returns>Value of the key (if any).</returns>
		internal static object GetValue(com.genesyslab.ail.ws.KeyValue[] keyValue, string key)
		{
			foreach(com.genesyslab.ail.ws.KeyValue kv in keyValue)
				if(kv.key == key)
					return kv.value;
			return null;
		}

		/// <summary>
		/// Gets the string value of a key in a key-value pair array.
		/// </summary>
		/// <param name="keyValue">Array of key-value pairs.</param>
		/// <param name="key">Key to search in the array.</param>
		/// <returns>String value of the key (if any).</returns>
		internal static string GetStringValue(com.genesyslab.ail.ws.KeyValue[] keyValue, string key)
		{
			foreach(com.genesyslab.ail.ws.KeyValue kv in keyValue)
				if(kv.key == key)
					return (kv.value != null ? (string)kv.value.ToString() : "");
			return null;
		}

		/// <summary>
		/// Releases the service factory.
		/// </summary>
		public void Disconnect()
		{
			this.mServiceFactory.releaseFactory();
		}
	}

	/// <summary>
	/// This class manages the services to handle an agent's place.
	/// </summary>
	public class Agent
	{
		public Connection mConnection;

		/// <summary>
		///  Agent properties.
		/// </summary>
		public string mAgentId;
		public string mAgentLogin;
		public string mAgentPassword;
		public string mQueue;
		public string mPlace;

		/// <summary>
		/// Attributes that AgentStatusForm uses to update.
		/// </summary>
		public VoiceMediaInfo[] mVoiceMediaInfo;
		public DnActionsPossible[] mDnActionsPossible;
		public MediaInfo[] mMediaInfo;
		public MediaActionsPossible[] mMediaActionsPossible;

		/// <summary>
		/// Attributes for event management.
		/// </summary>
		public bool listen = true;		
		private string mSubscriberId = null;
		private Notification mNotification = null;
		private ArrayList mEventsStack = null;

		/// <summary>
		/// Loads the Agent corresponding to the Agent properties and gets attribute values that correspond to mMediaInfo, mVoiceMediaInfo, ...
		/// </summary>
		/// <returns></returns>
		public bool Load()
		{
			// Getting the DTO for the agent.
			PersonDTO [] agentDTO = mConnection.mAgentService.getPersonsDTO(new string[] { mAgentId }, new string [] { "agent:dnsActionsPossible", "agent:mediasActionsPossible","agent:availableMedias"} );
			if(agentDTO != null && agentDTO.Length == 1)
			{
				PersonDTO mAgentDTO = agentDTO[0];

				// Getting the DTO for the agent's place.
				PlaceDTO[] placeDTO = mConnection.mPlaceService.getPlacesDTO(new string[] { mPlace }, new string[] { "place:dns","place:medias" });  
				if(placeDTO != null && placeDTO.Length == 1)
				{
					PlaceDTO mPlaceDTO = placeDTO[0];

					// Getting the DNs for the agent's place.
					object o = Connection.GetValue(mPlaceDTO.data, "place:dns");
					if(o != null)
					{
						mVoiceMediaInfo = (VoiceMediaInfo[])o;

						// Getting the possible agent actions on these DNs.
						o = Connection.GetValue(mAgentDTO.data, "agent:dnsActionsPossible");
						if(o != null)
							mDnActionsPossible = (DnActionsPossible[])o;						

					}
					o = Connection.GetValue(mPlaceDTO.data, "place:medias");
					//If the agent is logged in, media exist in the interaction server
					if(o != null)
					{
						mMediaInfo = (MediaInfo[])o;
						if(mMediaInfo.Length != 0) 
						{
							// Getting the possible agent actions on these DNs.
							o = Connection.GetValue(mAgentDTO.data, "agent:mediasActionsPossible");
							if(o != null)
							{
								mMediaActionsPossible = (MediaActionsPossible[])o;		
							}
						}
						else 
						{
							// Media info is not available in the place service
							// Creating MediaInfo and MediaActionsPossible arrays
							o = Connection.GetValue(mAgentDTO.data, "agent:availableMedias");
							string[] mediaNames = (string[])o;
							mMediaInfo = new MediaInfo[mediaNames.Length];
							mMediaActionsPossible = new MediaActionsPossible[mediaNames.Length];
							int i=0;
							foreach(string mediaName in mediaNames )
							{
								mMediaInfo[i] = new MediaInfo();
								mMediaInfo[i].name = mediaName;
								mMediaInfo[i].status = MediaStatus.LOGGED_OUT;							
								mMediaActionsPossible[i] = new MediaActionsPossible();

								if(mediaName == "chat")
								{
									mMediaInfo[i].type = MediaType.CHAT;
									mMediaActionsPossible[i].mediaType =  MediaType.CHAT;
								} 
								else if (mediaName == "email")
								{
									mMediaInfo[i].type = MediaType.EMAIL;
									mMediaActionsPossible[i].mediaType =  MediaType.EMAIL;
								}
								mMediaActionsPossible[i].agentActions = new AgentMediaAction[]{ AgentMediaAction.LOGIN };
								i++;
							}
						}
					}

					return true;
				}
			}
			return false;
		}

		public void setNotification(AgentStatusForm notifEndPoint)
		{
			if(notifEndPoint!=null)
			{
				listen = false;
				mNotification = new Notification();
				mNotification.notificationEndpoint = notifEndPoint;
				
				if(this.mConnection.mServiceFactory.ServiceFactoryImpl is com.genesyslab.ail.WebServicesFactory)
					mNotification.notificationType="SOAP_HTTP";
				else
					mNotification.notificationType="JAVA";
			}
			else 
			{
				mNotification=null;
				listen = true;
			}
			
			mConnection.mEventService.setNotification(this.mSubscriberId, mNotification);


		}

		/// <summary>
		/// Registers for voice media and media events occuring on the place.
		/// </summary>
		public string RegisterEvent()
		{			
			string msgResult = "";

			/// Creating the topics

			TopicsService [] topicServices = new TopicsService[1];
			topicServices[0] = new TopicsService();
			topicServices[0].serviceName = "AgentService";
		
			topicServices[0].topicsEvents = new TopicsEvent[2];
			topicServices[0].topicsEvents[0] = new TopicsEvent();
			topicServices[0].topicsEvents[0].eventName = "VoiceMediaEvent";
			topicServices[0].topicsEvents[0].attributes = new String[]{"agent:voiceMediaInfo","agent:dnActionsPossible"};
			topicServices[0].topicsEvents[0].triggers = new Topic[1];
			topicServices[0].topicsEvents[0].triggers[0] = new Topic();
			topicServices[0].topicsEvents[0].triggers[0].key = "PLACE";
			topicServices[0].topicsEvents[0].triggers[0].value = mPlace;
			topicServices[0].topicsEvents[0].filters = null;
			
			topicServices[0].topicsEvents[1] = new TopicsEvent();
			topicServices[0].topicsEvents[1].eventName = "MediaEvent";
			topicServices[0].topicsEvents[1].attributes = new String[]{"agent:mediaInfo","agent:mediaActionsPossible"};
			topicServices[0].topicsEvents[1].triggers = new Topic[1];
			topicServices[0].topicsEvents[1].triggers[0] = new Topic();
			topicServices[0].topicsEvents[1].triggers[0].key = "PLACE";
			topicServices[0].topicsEvents[1].triggers[0].value = mPlace;
			topicServices[0].topicsEvents[1].filters = null;

			/// Creating a subscriber if none
			if(this.mSubscriberId == null)
			{
				SubscriberResult result =
					mConnection.mEventService.createSubscriber(mNotification, topicServices);
			
				if(result.errors == null || result.errors.Length == 0)
				{
					mSubscriberId = result.subscriberId;
					mEventsStack = new ArrayList();					
				} 
				else
				{
					msgResult = "Error at subscriber creation";
				}
			}
			else
			{	/// Removing previous topics
				this.mConnection.mEventService.unsubscribeAllTopics(this.mSubscriberId);
	        
				/// Subscribing for these topics
				TopicServiceError[] topicsError
					= this.mConnection.mEventService.subscribeTopics(this.mSubscriberId,topicServices);

				if(topicsError.Length != 0)
				{
					foreach(TopicServiceError topicErr in topicsError)
					{
						msgResult = topicErr.error + " " +topicErr.serviceName + ": " + topicErr.eventName
							+ " for "+ topicErr.trigger.ToString() + " - "+topicErr.filter.ToString();
					}
				} 
			}
			return msgResult;
		}

		/// <summary>
		/// Unregisters.
		/// </summary>
		public void UnRegisterEvent()
		{
			listen = false;
			if(mSubscriberId != "")
				mConnection.mEventService.removeSubscriber(mSubscriberId);
			mSubscriberId = "";
			mEventsStack = null;			
		}
		/// <summary>
		/// Pulls events and adds them to the event stack.
		/// </summary>
		/// <returns>true if new events were Pulled.</returns>
		
		public bool HasEvent()
		{
			try
			{
				com.genesyslab.ail.ws._event.Event [] eventResult 
					= mConnection.mEventService.getEvents(mSubscriberId, 0);
				if(eventResult != null && eventResult.Length > 0)
				{
					mEventsStack.AddRange(eventResult);
					return true;
				}
				else return false;
			}
			catch(Exception e)
			{
				return false;
			}
		}

		/// <summary>
		/// Updates this object depending on the Pulled events.
		/// </summary>
		public void ProcessEvents()
		{
			/// Managing events if any.
			if(mEventsStack.Count > 0)
			{
				ArrayList eventsStack = new ArrayList(mEventsStack);
				foreach(com.genesyslab.ail.ws._event.Event e in eventsStack)
				{
					/// Removing an event from the stack
					mEventsStack.Remove(e);

					/// Managing the event
					HandleEvent(e);
				}

			}
		}

		/// <summary>
		/// Handles an event. This method is called each time that this Agent gets an Event from mConnection.mEventService.
		/// </summary>
		/// <param name="e"></param>
		public void HandleEvent(com.genesyslab.ail.ws._event.Event e)
		{
			/// Managing this event
			switch(e.eventName)
			{
				case "VoiceMediaEvent":
				{
					/// Updating the voice media info.
					object o = Connection.GetValue(e.attributes, "agent:voiceMediaInfo");
					if(o != null)
						mVoiceMediaInfo[0] = (VoiceMediaInfo)o;
											
					/// Updating the possible agent actions on DNs
					o = Connection.GetValue(e.attributes, "agent:dnActionsPossible");
					if(o != null)
					{
						if(mDnActionsPossible == null)
							mDnActionsPossible = new DnActionsPossible[mVoiceMediaInfo.Length];
						mDnActionsPossible[0] = (DnActionsPossible)o;
					}
				}
					break;
				case "MediaEvent":
				{
					/// Updating the voice media info.
					int index = 0;
					object o = Connection.GetValue(e.attributes, "agent:mediaInfo");
					if(o != null)
					{
						
						MediaInfo valInfo = (MediaInfo) o;
						index = getMediaIndex(valInfo);
						if (index< mMediaInfo.Length)
						{
							mMediaInfo[index] = valInfo;						
						}
					}

					/// Updating the possible agent actions on Medias
					o = Connection.GetValue(e.attributes, "agent:mediaActionsPossible");
					if(o != null)
					{
							index = getMediaActionsIndex((MediaActionsPossible)o);
							if(index<mMediaActionsPossible.Length)
								mMediaActionsPossible[index]=(MediaActionsPossible)o;
					}
				}
					break;
					
			}
		}

		/// <summary>
		/// Gets the index of a media in the mMediaInfo array.
		/// </summary>
		/// <param name="info"></param>
		/// <returns></returns>
		public int getMediaIndex(MediaInfo info)
		{
			int i=0;
			if(mMediaInfo!=null)
			{
				foreach(MediaInfo v in mMediaInfo )
				{
					if(v.name == info.name)
						return i;
					else i++;
				}
			}
			return i;
		}
		/// <summary>
		/// Gets the index of a MediaActionsPossible object in the mMediaActionsPossible array.
		/// </summary>
		/// <param name="info"></param>
		/// <returns></returns>
		public int getMediaActionsIndex(MediaActionsPossible info)
		{
			int i=0;
			if(mMediaActionsPossible != null)
			{
				foreach(MediaActionsPossible v in mMediaActionsPossible )
				{
				
					if(v.mediaType == info.mediaType)
						return i;
					else i++;
				}
			}
			return i;

		}
	
		/// <summary>
		/// Login this agent's DNs and medias.
		/// </summary>
		public void Login()
		{
			LoginVoiceForm myVoiceForm = new LoginVoiceForm();
			ArrayList alDn = new ArrayList();
			foreach(VoiceMediaInfo vmi in mVoiceMediaInfo)
				alDn.Add(vmi.dnId);
			myVoiceForm.dnIds = (string[])alDn.ToArray(typeof(string));
			myVoiceForm.loginId = mAgentLogin;
			myVoiceForm.password = mAgentPassword;
			myVoiceForm.queue = mQueue;
			myVoiceForm.workmode = com.genesyslab.ail.ws.agent.WorkmodeType.MANUAL_IN;
			myVoiceForm.reasons = null;
			myVoiceForm.TExtensions = null;
			MediaForm myMediaForm = new MediaForm();
			myMediaForm.reasonDescription ="Login on all media.";
			mConnection.mAgentService.login(mAgentId, mPlace, myVoiceForm, myMediaForm);
		}

		/// <summary>
		/// Logout this agent's DNs and media. 
		/// </summary>
		public void Logout()
		{
			LogoutVoiceForm myVoiceForm = new LogoutVoiceForm();
			ArrayList alDn = new ArrayList();
			foreach(VoiceMediaInfo vmi in mVoiceMediaInfo)
				alDn.Add(vmi.dnId);
			myVoiceForm.dnIds = (string[])alDn.ToArray(typeof(string));
			myVoiceForm.queue = mQueue;
			myVoiceForm.reasons = null;
			myVoiceForm.TExtensions = null;

			MediaForm myMediaForm = new MediaForm();
			myMediaForm.reasonDescription ="Logout on all media.";

			mConnection.mAgentService.logout(mAgentId, myVoiceForm, myMediaForm);
		}

		/// <summary>
		/// Sets this agent's DNs and media to ready. 
		/// </summary>
		public void Ready()
		{
			ReadyVoiceForm myVoiceForm = new ReadyVoiceForm();
			ArrayList alDn = new ArrayList();
			foreach(VoiceMediaInfo vmi in mVoiceMediaInfo)
				alDn.Add(vmi.dnId);
			myVoiceForm.dnIds = (string[])alDn.ToArray(typeof(string));
			myVoiceForm.queue = mQueue;
			myVoiceForm.workmode = com.genesyslab.ail.ws.agent.WorkmodeType.MANUAL_IN;
			myVoiceForm.reasons = null;
			myVoiceForm.TExtensions = null;

			MediaForm myMediaForm = new MediaForm();
			myMediaForm.reasonDescription ="Ready on all media.";

			mConnection.mAgentService.ready(mAgentId, myVoiceForm, myMediaForm);
		}

		/// <summary>
		/// Sets this agent's DNs and media to not-ready. 
		/// </summary>
		public void NotReady()
		{
			ReadyVoiceForm myVoiceForm = new ReadyVoiceForm();
			ArrayList alDn = new ArrayList();
			foreach(VoiceMediaInfo vmi in mVoiceMediaInfo)
				alDn.Add(vmi.dnId);
			myVoiceForm.dnIds = (string[])alDn.ToArray(typeof(string));
			myVoiceForm.queue = mQueue;
			myVoiceForm.workmode = com.genesyslab.ail.ws.agent.WorkmodeType.MANUAL_IN;
			myVoiceForm.reasons = null;
			myVoiceForm.TExtensions = null;
			
			MediaForm myMediaForm = new MediaForm();
			myMediaForm.reasonDescription ="Not Ready on all media.";

			mConnection.mAgentService.notReady(mAgentId, myVoiceForm, myMediaForm);
		}

	}
												 
}

