/*
 * Decompiled with CFR 0.152.
 */
package com.genesyslab.platform.commons.connection.tls;

import com.genesyslab.platform.commons.connection.tls.DummyPasswordCallbackHandler;
import com.genesyslab.platform.commons.connection.tls.KeyManagerHelper;
import com.genesyslab.platform.commons.connection.tls.SSLContextHelper;
import com.genesyslab.platform.commons.connection.tls.SSLExtendedOptions;
import com.genesyslab.platform.commons.connection.tls.TLSConfiguration;
import com.genesyslab.platform.commons.connection.tls.ThumbprintUtils;
import com.genesyslab.platform.commons.connection.tls.TrustManagerHelper;
import com.genesyslab.platform.commons.log.ILogger;
import com.genesyslab.platform.commons.log.Log;
import java.io.File;
import java.security.GeneralSecurityException;
import javax.net.ssl.SSLContext;
import javax.net.ssl.X509ExtendedKeyManager;
import javax.net.ssl.X509TrustManager;

public class TLSConfigurationHelper {
    private static final ILogger log = Log.getLogger(TLSConfigurationHelper.class);

    public static SSLContext createSslContext(TLSConfiguration config) throws SecurityException {
        SSLContext sslContext;
        if (null == config) {
            throw new SecurityException("config must not be null");
        }
        ProviderType prov = TLSConfigurationHelper.detectProvider(config);
        if (null == prov) {
            throw new SecurityException("Could not detect security provider type: TLS parameters are inconclusive or ambiguous");
        }
        TLSConfigurationHelper.validateTlsConfiguration(config, prov);
        X509ExtendedKeyManager km = null;
        try {
            switch (prov) {
                case PEM_FILE: {
                    km = KeyManagerHelper.createPEMKeyManager(config.getCertificate(), config.getCertificateKey());
                    break;
                }
                case MSCAPI: {
                    km = KeyManagerHelper.createMSCAPIKeyManager(config.getKeyStoreCallbackHandler(), config.getCertificate());
                    break;
                }
                case PKCS11: {
                    km = KeyManagerHelper.createPKCS11KeyManager(config.getKeyStoreCallbackHandler());
                }
            }
        }
        catch (Exception e) {
            throw new SecurityException("Could not create key manager based on TLS parameters", e);
        }
        X509TrustManager tm = null;
        try {
            String hostName = config.isTargetNameCheckEnabled() ? config.getExpectedHostname() : null;
            switch (prov) {
                case PEM_FILE: {
                    tm = TrustManagerHelper.createPEMTrustManager(config.getTrustedCaCertificate(), config.getCrl(), hostName);
                    break;
                }
                case MSCAPI: {
                    tm = TrustManagerHelper.createMSCAPITrustManager(config.getTrustStoreCallbackHandler(), config.getCrl(), hostName);
                    break;
                }
                case PKCS11: {
                    tm = TrustManagerHelper.createPKCS11TrustManager(config.getTrustStoreCallbackHandler(), config.getCrl());
                }
            }
        }
        catch (Exception e) {
            throw new SecurityException("Could not create trust manager based on TLS parameters", e);
        }
        try {
            sslContext = SSLContextHelper.createSSLContext(km, tm);
        }
        catch (GeneralSecurityException e) {
            throw new SecurityException("Could not create SSLContext based on TLS parameters", e);
        }
        return sslContext;
    }

    public static SSLExtendedOptions createSslExtendedOptions(TLSConfiguration config) {
        return new SSLExtendedOptions(config.isMutual(), config.getCipherList());
    }

    static ProviderType detectProvider(TLSConfiguration config) {
        log.debug((Object)"Detecting TLS security provider type...");
        if (null == config) {
            throw new IllegalArgumentException("config must not be null");
        }
        if (null != config.getProvider()) {
            if ("PEM".equalsIgnoreCase(config.getProvider())) {
                log.debug((Object)"Security provider is explicitly set to PEM.");
                return ProviderType.PEM_FILE;
            }
            if ("MSCAPI".equalsIgnoreCase(config.getProvider())) {
                log.debug((Object)"Security provider is explicitly set to MSCAPI.");
                return ProviderType.MSCAPI;
            }
            if ("PKCS11".equalsIgnoreCase(config.getProvider())) {
                log.debug((Object)"Security provider is explicitly set to PKCS11.");
                return ProviderType.PKCS11;
            }
            log.error((Object)("Security provider is set explicitly to " + config.getProvider() + ", but this type is not supported."));
            return null;
        }
        if (config.isFips140Enabled()) {
            log.debug((Object)"FIPS140 is enabled, implying PKCS11 provider.");
            return ProviderType.PKCS11;
        }
        if (null != config.getCertificate()) {
            if (ThumbprintUtils.isThumbprint(config.getCertificate())) {
                log.debug((Object)"Certificate value is a valid thumbprint, implying MSCAPI provider.");
                return ProviderType.MSCAPI;
            }
            log.debug((Object)"Certificate value is not a valid thumbprint.");
            if (TLSConfigurationHelper.isExistingFile(config.getCertificate())) {
                log.debug((Object)"Certificate value is a valid file path, implying PEM provider.");
                return ProviderType.PEM_FILE;
            }
            log.debug((Object)"Certificate value is not a path to an existing file.");
        } else {
            log.debug((Object)"Certificate value is null.");
        }
        if (null != config.getTrustedCaCertificate()) {
            if (ThumbprintUtils.isThumbprint(config.getTrustedCaCertificate())) {
                log.debug((Object)"Trusted CA value is a valid thumbprint, implying MSCAPI provider.");
                return ProviderType.MSCAPI;
            }
            log.debug((Object)"Trusted CA value value is not a valid thumbprint.");
            if (TLSConfigurationHelper.isExistingFile(config.getTrustedCaCertificate())) {
                log.debug((Object)"Trusted CA value is a valid file path, implying PEM provider.");
                return ProviderType.PEM_FILE;
            }
            log.debug((Object)"Trusted CA value is not a path to an existing file.");
        } else {
            log.debug((Object)"Trusted CA value is null.");
        }
        log.debug((Object)"Could not detect security provider.");
        return null;
    }

    private static void validateTlsConfiguration(TLSConfiguration config, ProviderType providerType) throws SecurityException {
        if (null == config) {
            throw new IllegalArgumentException("config must not be null");
        }
        if (null == providerType) {
            throw new IllegalArgumentException("providerType must not be null");
        }
        switch (providerType) {
            case PEM_FILE: {
                TLSConfigurationHelper.validatePemConfiguration(config);
                break;
            }
            case MSCAPI: {
                TLSConfigurationHelper.validateMscapiConfiguration(config);
                break;
            }
            case PKCS11: {
                TLSConfigurationHelper.validatePkcs11Configuration(config);
            }
        }
    }

    private static void validatePemConfiguration(TLSConfiguration config) throws SecurityException {
        log.debug((Object)"Validating PEM provider parameters...");
        if (config.isClientMode()) {
            if (!TLSConfigurationHelper.isExistingFile(config.getTrustedCaCertificate())) {
                throw new SecurityException("Could not find trusted CA certificate file by specified path. It is required for client application.");
            }
        } else {
            if (!TLSConfigurationHelper.isExistingFile(config.getCertificate())) {
                throw new SecurityException("Could not find certificate file by specified path. It is required for server application.");
            }
            if (!TLSConfigurationHelper.isExistingFile(config.getCertificateKey())) {
                throw new SecurityException("Could not find private key file by specified path. It is required for server application.");
            }
        }
        if (null != config.getCrl() && !TLSConfigurationHelper.isExistingFile(config.getCrl())) {
            throw new SecurityException("Could not find CRL file by specified path.");
        }
        if (config.isFips140Enabled()) {
            throw new SecurityException("fips140 configuration flag must be disabled for PEM file-based security provider");
        }
        log.debug((Object)"PEM provider parameters seem to be valid.");
    }

    private static void validateMscapiConfiguration(TLSConfiguration config) throws SecurityException {
        log.debug((Object)"Validating MSCAPI provider parameters...");
        if (null == config.getKeyStoreCallbackHandler()) {
            log.debug((Object)"Key store callback handler is null, using dummy instead...");
            config.setKeyStoreCallbackHandler(new DummyPasswordCallbackHandler());
        }
        if (null == config.getTrustStoreCallbackHandler()) {
            log.debug((Object)"Trust store callback handler is null, using dummy instead...");
            config.setTrustStoreCallbackHandler(new DummyPasswordCallbackHandler());
        }
        if (null != config.getCrl() && !TLSConfigurationHelper.isExistingFile(config.getCrl())) {
            throw new SecurityException("Could not find CRL file by specified path.");
        }
        if (config.isFips140Enabled()) {
            throw new SecurityException("fips140 configuration flag must be disabled for MSCAPI security provider");
        }
        log.debug((Object)"MSCAPI provider parameters seem to be valid.");
    }

    private static void validatePkcs11Configuration(TLSConfiguration config) throws SecurityException {
        log.debug((Object)"Validating PKCS11 provider parameters...");
        if (null == config.getKeyStoreCallbackHandler()) {
            throw new SecurityException("Key store callback handler must be set for PKCS11 security provider.");
        }
        if (null == config.getTrustStoreCallbackHandler()) {
            throw new SecurityException("Trust store callback handler must be set for PKCS11 security provider.");
        }
        if (null != config.getCrl() && !TLSConfigurationHelper.isExistingFile(config.getCrl())) {
            throw new SecurityException("Could not find CRL file by specified path.");
        }
        if (!config.isFips140Enabled()) {
            throw new SecurityException("fips140 configuration flag must be enabled for PKCS11 security provider.");
        }
        if (config.isTargetNameCheckEnabled()) {
            throw new SecurityException("Target name check flag is not supported for PKCS11 security provider.");
        }
        log.debug((Object)"PKCS11 provider parameters seem to be valid.");
    }

    static boolean isExistingFile(String s) {
        log.debug((Object)("Resolving file \"" + s + "\", current directory is \"" + System.getProperty("user.dir") + "\"..."));
        if (null == s || s.length() == 0) {
            log.debug((Object)"File path is null or empty");
            return false;
        }
        File target = new File(s);
        if (target.exists()) {
            log.debug((Object)("File \"" + s + "\" is found."));
            return true;
        }
        log.debug((Object)("Could not find file \"" + s + "\"."));
        return false;
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static enum ProviderType {
        PEM_FILE,
        MSCAPI,
        PKCS11;

    }
}

