﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Linq;
using System.Text;
using Genesyslab.Platform.ApplicationBlocks.ConfigurationObjectModel;
using Genesyslab.Platform.ApplicationBlocks.ConfigurationObjectModel.CfgObjects;
using Genesyslab.Platform.ApplicationBlocks.ConfigurationObjectModel.Queries;
using Genesyslab.Platform.Commons.Protocols;
using Genesyslab.Platform.Configuration.Protocols;
using Genesyslab.Platform.Configuration.Protocols.Types;

namespace SampleServerApp
{
    /// <summary>
    /// This class is used to access configuration server data. 
    /// </summary>
    class ConfigurationAccessor
    {
        private string host;
        private int port;
        private string userName;
        private string password;
        private string clientName;
        private CfgAppType clientType;
        IConfService confService;


        #region Internal

        /// <summary>
        /// Creates a new instance of the Configuration Server protocol
        /// and sets all of the attributes required for its use.
        /// </summary>
        /// <returns></returns>
        private ConfServerProtocol GetProtocol()
        {
            Endpoint desc = new Endpoint(host, port);

            ConfServerProtocol confProtocol = new ConfServerProtocol(desc);

            confProtocol.ClientApplicationType = (int)clientType;
            confProtocol.ClientName = clientName;
            confProtocol.UserName = userName;
            confProtocol.UserPassword = password;

            return confProtocol;
        }

        #endregion

        public ConfigurationAccessor()
        {
        }

        /// <summary>
        /// Returns the current state of the configuration server channel.  Should be
        /// used to determine whether a connection is opened.
        /// </summary>
        public ChannelState ChannelState
        {
            get
            {
                ConfServerProtocol protocol =
                    (ConfServerProtocol)confService.Protocol;

                return protocol == null ?
                    ChannelState.Closed :
                    protocol.State;
            }
        }

   

        /// <summary>
        /// Used to open the connection to the Configuration Server specified
        /// in the App.config file for this application
        /// </summary>
        public void Connect()
        {
            ConfServerProtocol protocol = GetProtocol();

            try
            {
                protocol.Open();
            }
            catch (Exception ex)
            {
                throw new InvalidOperationException("Cannot connect to the configuration server (" + ex.Message + ").");
            }

            confService = ConfServiceFactory.CreateConfService(protocol);
        }

        /// <summary>
        /// Should be called before working with Configuration Server.  This method
        /// reads the necessary configuration information from the App.config and 
        /// sets the appropriate properties.
        /// </summary>
        public void Initialize()
        {
            host = ConfigurationManager.AppSettings["ConfigServerHost"];
            try
            {
                port = int.Parse(ConfigurationManager.AppSettings["ConfigServerPort"]);
            }
            catch 
            {
                throw new Exception("Invalid Configuration Server port");
            }
           
            userName = ConfigurationManager.AppSettings["UserName"];
            password = ConfigurationManager.AppSettings["Password"];
            clientName = ConfigurationManager.AppSettings["ClientName"];

            clientType = CfgAppType.CFGGenericServer;

        }

        /// <summary>
        /// Closes connection with configuration server
        /// </summary>
        internal void Disconnect()
        {
            if (confService != null)
            {
                IProtocol protocol = confService.Protocol;
                
                ConfServiceFactory.ReleaseConfService(confService);

                if (protocol != null && protocol.State!=ChannelState.Closed)
                    protocol.Close();
            }

        }

        /// <summary>
        /// Retrieves an application by name.
        /// </summary>
        /// <param name="appName">application name as seen in configuration server</param>
        /// <returns>CfgApplication object describing the retrieved application, or null if not found</returns>
        internal CfgApplication RetrieveApplication(string appName)
        {
            CfgApplicationQuery query = new CfgApplicationQuery();

            query.Name = appName;

            return confService.RetrieveObject<CfgApplication>(query);
        }
    }

}
