﻿using System;
using System.Configuration;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using Genesyslab.Platform.ApplicationBlocks.ConfigurationObjectModel.CfgObjects;
using Genesyslab.Platform.Configuration.Protocols.Types;
using Genesyslab.Platform.Management.Protocols;
using Genesyslab.Platform.Management.Protocols.LocalControlAgent.Events;

namespace SampleServerApp
{
    class SampleServerApp : AbstractServerApp
    {

        protected static void Log(String message)
        {
            String text = DateTime.Now.ToString("HH:mm:ss") + " [" + Thread.CurrentThread.Name + "]" + ": " + message;
            Console.WriteLine(text);
            File.AppendAllText("ApplicationLog.txt", text+"\n");
        }

        public static void Main(String[] args)
        {
            // Retrieve application from Configuration Server
            CfgApplication appConfiguration = RetrieveApplicationConfiguration();

            int lcaPort = ReadLCAPort();

            SampleServerApp theApp = new SampleServerApp();
            theApp.OpenLCA(lcaPort, (int)appConfiguration.Type, appConfiguration.Name, appConfiguration.DBID);

            Log("Main: LCA opened");

            // ...
            Thread.Sleep(1500);
            Log("Main: app started");

            // Then application should report that initialization is finished:
            theApp.DoUpdateStatus(theApp.ExecutionMode, ApplicationStatus.Running);
            Log("Main: app is \"running\"");
            // After this SCS through LCA may send request to change execution mode.
            // Initially applications start as "Primary", but when particular application
            // is configured in primary/backup pair, and 

            while (theApp.ExecutionMode != ApplicationExecutionMode.Exiting)
            {
                Thread.Sleep(500);
            }

            Log("Main: app is stopping");
            // Report application stopped status:
            theApp.DoUpdateStatus(theApp.ExecutionMode, ApplicationStatus.Stopped);


            theApp.CloseLCA();
            Log("Main: LCA closed");

            
        }

     /// <summary>
     /// This method opens connection to Configuration Server, reads application configuration and then closes connection.
     /// The <see cref="CfgApplication"/> contains application DBID required for communication with LCA.     
     /// </summary>
     /// <returns></returns>
        
        private static CfgApplication RetrieveApplicationConfiguration()
        {
            ConfigurationAccessor accessor = new ConfigurationAccessor();  
            accessor.Initialize();            
            accessor.Connect();

 
            CfgApplication application = null;
            try
            {
                String clientName = ConfigurationManager.AppSettings["ClientName"];
                application = accessor.RetrieveApplication(clientName);

                if (application == null)
                {
                    throw new Exception("Specified application not found!");
                }
            }
            catch (Exception e)
            {
                throw new Exception( "Can't retrieve application! Reason: " + e.Message);
            }
            finally
            {
                accessor.Disconnect();
            }
            Log(String.Format("Got application setting from config server: \n Name={0}, \n DBID={1}", application.Name, application.DBID));
            return application;
        }

        private static int ReadLCAPort()
        {
            int LCAPort = 0;
            try
            {
                LCAPort = int.Parse(ConfigurationManager.AppSettings["LCAPort"]);
            }
            catch
            {
                throw new Exception("Invalid LCA port!");               
            }
            return LCAPort;
        }
        

        protected override void OnLCAConnectionOpened()
        {
            Log("LCA connected");
        }

        protected override void OnLCAConnectionLost(EventArgs message)
        {
            Log("LCA lost");
        }

        protected override void OnHostUnderSCSControl(EventHostUnderControl message)
        {
            Log("Under SCS control (" + message.SCSApplicationName + ")");
        }

        protected override void OnHostOffSCSControl(EventNoControllingSCS message)
        {
            Log("No SCS control");
        }

        protected override bool OnChangeExecutionMode(ApplicationExecutionMode execMode)
        {
            Log("Changing execution mode to " + execMode + "...");
            return true;
        }

        protected override void OnSuspendApplication()
        {
            Log("Do application suspend...");
        }
    }
}
